#include <gb.h>
#include <drawing.h>

#define NB_TOOLS 18
#define NB_DATA_TILES 48

#define ARROW 0
#define PEN 1
#define BRUSH 2
#define FILL 3
#define ERASER 4
#define CROSS 5

#define UNSELECTED 0
#define SELECTED 1

#define FIRST_TOOL 0
#define LAST_TOOL 9
#define FIRST_COLOR 10
#define LAST_COLOR 13
#define FIRST_MODE 14
#define LAST_MODE 17

typedef struct icon_info_
{
  UBYTE data_idx; /* Index of data in the data array */
  UBYTE x, y, w, h; /* Position and size (in tiles) */
  UBYTE up, down, left, right; /* Index of next icons (for cursor movements) */
  UBYTE cursor; /* Cursor associated with icon */
} icon_info;

typedef struct cursor_info_
{
  UBYTE data_idx; /* Index of data in the data array */
  UBYTE w, h; /* Size (in tiles) */
  UBYTE hot_x, hot_y; /* Position of hot point, relatice to top-left of sprite (in pixels) */
} cursor_info;

icon_info icons[] =
{
  /* Pen */
  { 0, 0, 0, 2, 2, 10, 2, 1, 1, 1 }, /* 0 */
  /* Brush */
  { 4, 2, 0, 2, 2, 14, 3, 0, 0, 2 }, /* 1 */
  /* Fill */
  { 8, 0, 2, 2, 2, 0, 4, 3, 3, 3 }, /* 2 */
  /* Eraser */
  { 12, 2, 2, 2, 2, 1, 5, 2, 2, 4 }, /* 3 */
  /* Line */
  { 16, 0, 4, 2, 2, 2, 6, 5, 5, 5 }, /* 4 */
  /* Line */
  { 20, 2, 4, 2, 2, 3, 7, 4, 4, 5 }, /* 5 */
  /* Rectangle */
  { 24, 0, 6, 2, 2, 4, 8, 7, 7, 5 }, /* 6 */
  /* Oval */
  { 28, 2, 6, 2, 2, 5, 9, 6, 6, 5 }, /* 7 */
  /* Filled rectangle */
  { 32, 0, 8, 2, 2, 6, 10, 9, 9, 5 }, /* 8 */
  /* Filled oval */
  { 36, 2, 8, 2, 2, 7, 14, 8, 8, 5 }, /* 9 */
  /* Color */
  { 40, 0, 10, 1, 1, 8, 11, 16, 12, 0 }, /* 10 */
  { 41, 0, 11, 1, 1, 10, 0, 17, 13, 0 }, /* 11 */
  { 42, 1, 10, 1, 1, 8, 13, 10, 14, 0 }, /* 12 */
  { 43, 1, 11, 1, 1, 12, 0, 11, 15, 0 }, /* 13 */
  /* Mode */
  { 44, 2, 10, 1, 1, 9, 15, 12, 16, 0 }, /* 14 */
  { 45, 2, 11, 1, 1, 14, 1, 13, 17, 0 }, /* 15 */
  { 46, 3, 10, 1, 1, 9, 17, 14, 10, 0 }, /* 16 */
  { 47, 3, 11, 1, 1, 16, 1, 15, 11, 0 } /* 17 */
};

unsigned char data[NB_DATA_TILES][0x10] =
{
  /* Pen */
  { 0x00,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x01 },
  { 0x7E,0x02,0x7D,0x05,0x7B,0x0A,0x77,0x14,0x7F,0x18,0x7F,0x27,0x7F,0x00,0x00,0x7F },
  { 0x00,0x00,0xFE,0x01,0xFE,0x0D,0xF6,0x15,0xEE,0x29,0xDE,0x51,0xBE,0xA1,0x7E,0x41 },
  { 0xFE,0x81,0xFE,0x01,0xFE,0x01,0xFE,0x01,0xFE,0x01,0xFE,0xFD,0xFE,0x01,0x00,0xFF },
  /* Brush */
  { 0x00,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x01 },
  { 0x7E,0x02,0x7D,0x0D,0x7F,0x1E,0x7F,0x1E,0x7F,0x1D,0x7F,0x11,0x7F,0x00,0x00,0x7F },
  { 0x00,0x00,0xFE,0x01,0xFE,0x0D,0xF6,0x15,0xEE,0x29,0xDE,0x51,0xBE,0xA1,0x7E,0x41 },
  { 0xFE,0x81,0xFE,0x01,0xFE,0x01,0xFE,0x01,0xFE,0xFD,0xFE,0xFD,0xFE,0x01,0x00,0xFF },
  /* Fill */
  { 0x00,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x01,0x7F,0x02,0x7E,0x0C,0x7D,0x18,0x7E,0x1D },
  { 0x7F,0x1B,0x7F,0x19,0x7F,0x18,0x7F,0x08,0x7F,0x08,0x7F,0x08,0x7F,0x00,0x00,0x7F },
  { 0x00,0x00,0xFE,0x01,0xFE,0x81,0xFE,0x41,0x7E,0x21,0xBE,0x71,0x5E,0xE9,0xAE,0xC5 },
  { 0x5E,0x89,0xBE,0x11,0xFE,0xA1,0xFE,0x41,0xFE,0x01,0xFE,0x01,0xFE,0x01,0x00,0xFF },
  /* Eraser */
  { 0x00,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x01,0x7E,0x02,0x7C,0x04 },
  { 0x78,0x08,0x7F,0x1F,0x71,0x11,0x7F,0x1F,0x7F,0x00,0x7F,0x00,0x7F,0x00,0x00,0x7F },
  { 0x00,0x00,0xFE,0x01,0xFE,0x01,0xFE,0x01,0xFE,0xF9,0x1E,0x19,0x2E,0x29,0x5E,0x51 },
  { 0xBE,0xA1,0x7E,0x41,0xFE,0x81,0xFE,0x01,0xFE,0x01,0xFE,0x01,0xFE,0x01,0x00,0xFF },
  /* Line */
  { 0x00,0x00,0x7F,0x00,0x7F,0x20,0x7F,0x10,0x7F,0x08,0x7F,0x04,0x7F,0x02,0x7F,0x01 },
  { 0x7F,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x00,0x00,0x7F },
  { 0x00,0x00,0xFE,0x01,0xFE,0x01,0xFE,0x01,0xFE,0x01,0xFE,0x01,0xFE,0x01,0xFE,0x01 },
  { 0xFE,0x81,0xFE,0x41,0xFE,0x21,0xFE,0x11,0xFE,0x09,0xFE,0x05,0xFE,0x01,0x00,0xFF },
  /* Arc */
  { 0x00,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x1E,0x7F,0x01,0x7F,0x00,0x7F,0x00,0x7F,0x00 },
  { 0x7F,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x00,0x00,0x7F },
  { 0x00,0x00,0xFE,0x01,0xFE,0x01,0xFE,0x01,0xFE,0x81,0xFE,0x41,0xFE,0x21,0xFE,0x11 },
  { 0xFE,0x11,0xFE,0x09,0xFE,0x09,0xFE,0x09,0xFE,0x09,0xFE,0x01,0xFE,0x01,0x00,0xFF },
  /* Rectangle */
  { 0x00,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x3F,0x7F,0x20,0x7F,0x20,0x7F,0x20,0x7F,0x20 },
  { 0x7F,0x20,0x7F,0x20,0x7F,0x20,0x7F,0x20,0x7F,0x3F,0x7F,0x00,0x7F,0x00,0x00,0x7F },
  { 0x00,0x00,0xFE,0x01,0xFE,0x01,0xFE,0xFD,0xFE,0x05,0xFE,0x05,0xFE,0x05,0xFE,0x05 },
  { 0xFE,0x05,0xFE,0x05,0xFE,0x05,0xFE,0x05,0xFE,0xFD,0xFE,0x01,0xFE,0x01,0x00,0xFF },
  /* Oval */
  { 0x00,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x03,0x7F,0x0C,0x7F,0x10,0x7F,0x20,0x7F,0x20 },
  { 0x7F,0x20,0x7F,0x20,0x7F,0x10,0x7F,0x0C,0x7F,0x03,0x7F,0x00,0x7F,0x00,0x00,0x7F },
  { 0x00,0x00,0xFE,0x01,0xFE,0x01,0xFE,0xC1,0xFE,0x31,0xFE,0x09,0xFE,0x05,0xFE,0x05 },
  { 0xFE,0x05,0xFE,0x05,0xFE,0x09,0xFE,0x31,0xFE,0xC1,0xFE,0x01,0xFE,0x01,0x00,0xFF },
  /* Filled rectangle */
  { 0x00,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x3F,0x6A,0x3F,0x75,0x3F,0x6A,0x3F,0x75,0x3F },
  { 0x6A,0x3F,0x75,0x3F,0x6A,0x3F,0x75,0x3F,0x7F,0x3F,0x7F,0x00,0x7F,0x00,0x00,0x7F },
  { 0x00,0x00,0xFE,0x01,0xFE,0x01,0xFE,0xFD,0xAE,0xFD,0x56,0xFD,0xAE,0xFD,0x56,0xFD },
  { 0xAE,0xFD,0x56,0xFD,0xAE,0xFD,0x56,0xFD,0xFE,0xFD,0xFE,0x01,0xFE,0x01,0x00,0xFF },
  /* Filled oval */
  { 0x00,0x00,0x7F,0x00,0x7F,0x00,0x7F,0x03,0x7E,0x0F,0x75,0x1F,0x6A,0x3F,0x75,0x3F },
  { 0x6A,0x3F,0x75,0x3F,0x7A,0x1F,0x7D,0x0F,0x7F,0x03,0x7F,0x00,0x7F,0x00,0x00,0x7F },
  { 0x00,0x00,0xFE,0x01,0xFE,0x01,0xFE,0xC1,0xBE,0xF1,0x5E,0xF9,0xAE,0xFD,0x56,0xFD },
  { 0xAE,0xFD,0x56,0xFD,0xAE,0xF9,0x7E,0xF1,0xFE,0xC1,0xFE,0x01,0xFE,0x01,0x00,0xFF },
  /* Color */
  { 0x00,0x00,0x7F,0x7F,0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40,0x40 },
  { 0x40,0x7F,0x40,0x7F,0x40,0x7F,0x40,0x7F,0x40,0x7F,0x40,0x7F,0x7F,0x7F,0x00,0x7F },
  { 0x00,0x00,0xFE,0xFF,0xFE,0x03,0xFE,0x03,0xFE,0x03,0xFE,0x03,0xFE,0x03,0xFE,0x03 },
  { 0xFE,0xFF,0xFE,0xFF,0xFE,0xFF,0xFE,0xFF,0xFE,0xFF,0xFE,0xFF,0xFE,0xFF,0x00,0xFF },
  /* Mode */
  { 0x00,0x00,0x7F,0x00,0x7F,0x0C,0x7F,0x10,0x7F,0x08,0x7F,0x04,0x7F,0x18,0x7F,0x00 },
  { 0x7F,0x00,0x7F,0x08,0x7F,0x14,0x7F,0x14,0x7F,0x14,0x7F,0x08,0x7F,0x00,0x00,0x7F },
  { 0x00,0x01,0xFE,0x01,0xFE,0x29,0xFE,0x29,0xFE,0x11,0xFE,0x29,0xFE,0x29,0xFE,0x01 },
  { 0xFE,0x01,0xFE,0x11,0xFE,0x29,0xFE,0x39,0xFE,0x29,0xFE,0x29,0xFE,0x01,0x00,0xFF }
};

unsigned char selected_data[NB_DATA_TILES][0x10] =
{
  /* Pen */
  { 0xFF,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFE },
  { 0x81,0xFD,0x82,0xFA,0x84,0xF5,0x88,0xEB,0x80,0xE7,0x80,0xD8,0x80,0xFF,0xFF,0x80 },
  { 0xFF,0xFF,0x01,0xFE,0x01,0xF2,0x09,0xEA,0x11,0xD6,0x21,0xAE,0x41,0x5E,0x81,0xBE },
  { 0x01,0x7E,0x01,0xFE,0x01,0xFE,0x01,0xFE,0x01,0xFE,0x01,0x02,0x01,0xFE,0xFF,0x00 },
  /* Brush */
  { 0xFF,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFE },
  { 0x81,0xFD,0x82,0xF2,0x80,0xE1,0x80,0xE1,0x80,0xE2,0x80,0xEE,0x80,0xFF,0xFF,0x80 },
  { 0xFF,0xFF,0x01,0xFE,0x01,0xF2,0x09,0xEA,0x11,0xD6,0x21,0xAE,0x41,0x5E,0x81,0xBE },
  { 0x01,0x7E,0x01,0xFE,0x01,0xFE,0x01,0xFE,0x01,0x02,0x01,0x02,0x01,0xFE,0xFF,0x00 },
  /* Fill */
  { 0xFF,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFE,0x80,0xFD,0x81,0xF3,0x82,0xE7,0x81,0xE2 },
  { 0x80,0xE4,0x80,0xE7,0x80,0xE7,0x80,0xF7,0x80,0xF7,0x80,0xF7,0x80,0xFF,0xFF,0x80 },
  { 0xFF,0xFF,0x01,0xFE,0x01,0x7E,0x01,0xBE,0x81,0xDE,0x41,0x8E,0xA1,0x16,0x51,0x3A },
  { 0xA1,0x76,0x41,0xEE,0x01,0x5E,0x01,0xBE,0x01,0xFE,0x01,0xFE,0x01,0xFE,0xFF,0x00 },
  /* Eraser */
  { 0xFF,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFE,0x81,0xFD,0x83,0xFB },
  { 0x87,0xF7,0x80,0xE0,0x8E,0xEE,0x80,0xE0,0x80,0xFF,0x80,0xFF,0x80,0xFF,0xFF,0x80 },
  { 0xFF,0xFF,0x01,0xFE,0x01,0xFE,0x01,0xFE,0x01,0x06,0xE1,0xE6,0xD1,0xD6,0xA1,0xAE },
  { 0x41,0x5E,0x81,0xBE,0x01,0x7E,0x01,0xFE,0x01,0xFE,0x01,0xFE,0x01,0xFE,0xFF,0x00 },
  /* Line */
  { 0xFF,0xFF,0x80,0xFF,0x80,0xDF,0x80,0xEF,0x80,0xF7,0x80,0xFB,0x80,0xFD,0x80,0xFE },
  { 0x80,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFF,0xFF,0x80 },
  { 0xFF,0xFF,0x01,0xFE,0x01,0xFE,0x01,0xFE,0x01,0xFE,0x01,0xFE,0x01,0xFE,0x01,0xFE },
  { 0x01,0x7E,0x01,0xBE,0x01,0xDE,0x01,0xEE,0x01,0xF6,0x01,0xFA,0x01,0xFE,0xFF,0x00 },
  /* Arc */
  { 0xFF,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xE1,0x80,0xFE,0x80,0xFF,0x80,0xFF,0x80,0xFF },
  { 0x80,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFF,0xFF,0x80 },
  { 0xFF,0xFF,0x01,0xFE,0x01,0xFE,0x01,0xFE,0x01,0x7E,0x01,0xBE,0x01,0xDE,0x01,0xEE },
  { 0x01,0xEE,0x01,0xF6,0x01,0xF6,0x01,0xF6,0x01,0xF6,0x01,0xFE,0x01,0xFE,0xFF,0x00 },
  /* Rectangle */
  { 0xFF,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xC0,0x80,0xDF,0x80,0xDF,0x80,0xDF,0x80,0xDF },
  { 0x80,0xDF,0x80,0xDF,0x80,0xDF,0x80,0xDF,0x80,0xC0,0x80,0xFF,0x80,0xFF,0xFF,0x80 },
  { 0xFF,0xFF,0x01,0xFE,0x01,0xFE,0x01,0x02,0x01,0xFA,0x01,0xFA,0x01,0xFA,0x01,0xFA },
  { 0x01,0xFA,0x01,0xFA,0x01,0xFA,0x01,0xFA,0x01,0x02,0x01,0xFE,0x01,0xFE,0xFF,0x00 },
  /* Oval */
  { 0xFF,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFC,0x80,0xF3,0x80,0xEF,0x80,0xDF,0x80,0xDF },
  { 0x80,0xDF,0x80,0xDF,0x80,0xEF,0x80,0xF3,0x80,0xFC,0x80,0xFF,0x80,0xFF,0xFF,0x80 },
  { 0xFF,0xFF,0x01,0xFE,0x01,0xFE,0x01,0x3E,0x01,0xCE,0x01,0xF6,0x01,0xFA,0x01,0xFA },
  { 0x01,0xFA,0x01,0xFA,0x01,0xF6,0x01,0xCE,0x01,0x3E,0x01,0xFE,0x01,0xFE,0xFF,0x00 },
  /* Filled rectangle */
  { 0xFF,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xC0,0x9F,0xC0,0x9F,0xC0,0x9F,0xC0,0x9F,0xC0 },
  { 0x9F,0xC0,0x9F,0xC0,0x9F,0xC0,0x9F,0xC0,0x80,0xC0,0x80,0xFF,0x80,0xFF,0xFF,0x80 },
  { 0xFF,0xFF,0x01,0xFE,0x01,0xFE,0x01,0x02,0xF9,0x02,0xF9,0x02,0xF9,0x02,0xF9,0x02 },
  { 0xF9,0x02,0xF9,0x02,0xF9,0x02,0xF9,0x02,0x01,0x02,0x01,0xFE,0x01,0xFE,0xFF,0x00 },
  /* Filled oval */
  { 0xFF,0xFF,0x80,0xFF,0x80,0xFF,0x80,0xFC,0x83,0xF0,0x8F,0xE0,0x9F,0xC0,0x9F,0xC0 },
  { 0x9F,0xC0,0x9F,0xC0,0x8F,0xE0,0x83,0xF0,0x80,0xFC,0x80,0xFF,0x80,0xFF,0xFF,0x80 },
  { 0xFF,0xFF,0x01,0xFE,0x01,0xFE,0x01,0x3E,0xC1,0x0E,0xF1,0x06,0xF9,0x02,0xF9,0x02 },
  { 0xF9,0x02,0xF9,0x02,0xF1,0x06,0xC1,0x0E,0x01,0x3E,0x01,0xFE,0x01,0xFE,0xFF,0x00 },
  /* Color */
  { 0x00,0x00,0x7F,0x7F,0x61,0x61,0x52,0x52,0x4C,0x4C,0x4C,0x4C,0x52,0x52,0x61,0x61 },
  { 0x40,0x5E,0x40,0x6D,0x40,0x73,0x40,0x73,0x40,0x6D,0x40,0x5E,0x7F,0x7F,0x00,0x7F },
  { 0x00,0x00,0xFE,0xFF,0xFE,0x87,0xFE,0x4B,0xFE,0x33,0xFE,0x33,0xFE,0x4B,0xFE,0x87 },
  { 0x7A,0x7B,0xB6,0xB7,0xCE,0xCF,0xCE,0xCF,0xB6,0xB7,0x7A,0x7B,0xFE,0xFF,0x00,0xFF },
  /* Mode */
  { 0x00,0x00,0x7F,0x7F,0x73,0x73,0x6F,0x6F,0x77,0x77,0x7B,0x7B,0x67,0x67,0x7F,0x7F },
  { 0x7F,0x7F,0x77,0x77,0x6B,0x6B,0x6B,0x6B,0x6B,0x6B,0x77,0x77,0x7F,0x7F,0x00,0x7F },
  { 0x00,0x01,0xFE,0xFF,0xD6,0xD7,0xD6,0xD7,0xEE,0xEF,0xD6,0xD7,0xD6,0xD7,0xFE,0xFF },
  { 0xFE,0xFF,0xEE,0xEF,0xD6,0xD7,0xC6,0xC7,0xD6,0xD7,0xD6,0xD7,0xFE,0xFF,0x00,0xFF }
};

cursor_info cursors[] =
{
  /* Arrow */
  { 0, 1, 1, 0, 0 },
  /* Pen */
  { 1, 2, 2, 0, 15 },
  /* Brush */
  { 5, 2, 2, 0, 15 },
  /* Fill */
  { 9, 2, 2, 2, 15 },
  /* Eraser */
  { 13, 2, 2, 0, 15 },
  /* Cross */
  { 17, 2, 2, 5, 10 }
};

unsigned char cursors_data[][0x10] = {
  /* Arrow */
  { 0xFF,0x00,0xFF,0x7E,0xFF,0x7C,0xFE,0x78,0xFF,0x7C,0xFF,0x6E,0xFF,0x46,0xEF,0x00 },
  /* Pen */
  { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x03,0x00,0x07,0x01 },
  { 0x0F,0x02,0x1F,0x05,0x3F,0x0A,0x7F,0x14,0x7E,0x28,0xFC,0x30,0xF8,0x40,0x60,0x00 },
  { 0x00,0x00,0x00,0x00,0x3C,0x00,0x7C,0x18,0xFC,0x28,0xFC,0x50,0xF8,0xA0,0xF0,0x40 },
  { 0xE0,0x80,0xC0,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
  /* Brush */
  { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x03,0x00,0x07,0x01,0x0F,0x02 },
  { 0x1F,0x05,0x7F,0x0A,0xFF,0x34,0xFE,0x78,0xFC,0x78,0xFC,0x70,0xF8,0x40,0x60,0x00 },
  { 0x00,0x00,0x00,0x00,0x78,0x00,0xF8,0x30,0xF8,0x50,0xF8,0xA0,0xF0,0x40,0xE0,0x80 },
  { 0xC0,0x00,0x80,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
  /* Fill */
  { 0x00,0x00,0x00,0x00,0x07,0x00,0x0F,0x02,0x1F,0x05,0x7F,0x08,0xFE,0x31,0xFD,0x63 },
  { 0xFA,0x77,0xFD,0x6E,0xFF,0x64,0xFF,0x62,0xF7,0x21,0x73,0x20,0x70,0x20,0x50,0x00 },
  { 0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x00,0xC0,0x00,0xE0,0x80,0xF0,0xC0,0x78,0xA0 },
  { 0xF8,0x10,0xF8,0x20,0xF0,0x40,0xE0,0x80,0xC0,0x00,0x80,0x00,0x00,0x00,0x00,0x00 },
  /* Eraser */
  { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x07,0x00,0x0F,0x03 },
  { 0x1F,0x04,0x3F,0x08,0x7F,0x11,0xFF,0x22,0xFF,0x7D,0xFF,0x46,0xFF,0x7C,0x7E,0x00 },
  { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xF0,0x00,0xF0,0xE0 },
  { 0xF0,0x60,0xF0,0xA0,0xF0,0x40,0xE0,0x80,0xC0,0x00,0x80,0x00,0x00,0x00,0x00,0x00 },
  /* Cross */
  { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0E,0x00,0x0E,0x04,0x0E,0x04 },
  { 0x0E,0x04,0xFF,0x04,0xFB,0x7B,0xFF,0x04,0x0E,0x04,0x0E,0x04,0x0E,0x04,0x0E,0x00 },
  { 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
  { 0x00,0x00,0xE0,0x00,0xE0,0xC0,0xE0,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00 },
};

unsigned char data_buffer[NB_DATA_TILES][0x10];

UBYTE colors[] = { WHITE, DKGREY, LTGREY, BLACK };
UBYTE modes[] = { SOLID, OR, XOR, AND };

UBYTE current_tool;
UBYTE current_color;
UBYTE current_mode;
UBYTE current_cursor;
UBYTE menu_cursor_pos;

UBYTE cursor_x;
UBYTE cursor_y;

void set_cursor()
{
  UBYTE x, y, i;

  i = 0;
  for(x = 0; x < cursors[current_cursor].w; x++)
    for(y = 0; y < cursors[current_cursor].h; y++) {
      set_sprite_data(i,
		      1,
		      cursors_data[cursors[current_cursor].data_idx+i]);
      set_sprite_tile(i, i);
      set_sprite_prop(i, 0x10);
      i++;
    }
  /* Hide unused sprites */
  for(; i < 4; i++) {
    move_sprite(i, 0, 0);
  }
}

void move_cursor()
{
  UBYTE x, y;

  for(x = 0; x < cursors[current_cursor].w; x++)
    for(y = 0; y < cursors[current_cursor].h; y++)
      move_sprite((x<<1)+y,
		  cursor_x+8 - cursors[current_cursor].hot_x + (x<<3),
		  cursor_y+16 - cursors[current_cursor].hot_y + (y<<3));
}

void move_menu_cursor()
{
  move_sprite(0,
	      ((icons[menu_cursor_pos].x+icons[menu_cursor_pos].w)<<3) + 4,
	      ((icons[menu_cursor_pos].y+icons[menu_cursor_pos].h)<<3) + 12);
}

void set_icon(UBYTE icon, UBYTE selected)
{
  UBYTE x, y;

  for(x = 0; x < icons[icon].w; x++)
    for(y = 0; y < icons[icon].h; y++)
      switch_data(icons[icon].x + x,
		  icons[icon].y + y,
		  (selected ?
		   selected_data[icons[icon].data_idx+(x<<1)+y] :
		   data[icons[icon].data_idx+(x<<1)+y]),
		  data_buffer[icons[icon].data_idx+(x<<1)+y]);
}

void change_icon(UBYTE icon, UBYTE selected)
{
  UBYTE x, y;

  for(x = 0; x < icons[icon].w; x++)
    for(y = 0; y < icons[icon].h; y++)
      switch_data(icons[icon].x + x,
		  icons[icon].y + y,
		  (selected ?
		   selected_data[icons[icon].data_idx+(x<<1)+y] :
		   data[icons[icon].data_idx+(x<<1)+y]),
		  NULL);
}

void reset_icon(UBYTE icon)
{
  UBYTE x, y;

  for(x = 0; x < icons[icon].w; x++)
    for(y = 0; y < icons[icon].h; y++)
      switch_data(icons[icon].x + x,
		  icons[icon].y + y,
		  data_buffer[icons[icon].data_idx+(x<<1)+y],
		  NULL);
}

void splash()
{
  UBYTE x, y;

  cursor_x = 40;
  cursor_y = 50;
  current_cursor = PEN;
  set_cursor();
  move_cursor();
  SHOW_SPRITES;

  for(; cursor_x < 120; cursor_x++) {
    wait_vbl_done();
    move_cursor();
    plot(cursor_x, cursor_y, BLACK, SOLID);
  }
  for(; cursor_y < 94; cursor_y++) {
    wait_vbl_done();
    move_cursor();
    plot(cursor_x, cursor_y, BLACK, SOLID);
  }
  for(; cursor_x > 40; cursor_x--) {
    wait_vbl_done();
    move_cursor();
    plot(cursor_x, cursor_y, BLACK, SOLID);
  }
  for(; cursor_y > 50; cursor_y--) {
    wait_vbl_done();
    move_cursor();
    plot(cursor_x, cursor_y, BLACK, SOLID);
  }
  cursor_x = 160/2;
  cursor_y = 144/2;
  current_cursor = FILL;
  set_cursor();
  move_cursor();

  for(y = 51; y < 94; y++)
    for(x = 41; x < 120; x++)
      plot(x, y, LTGREY, SOLID);

  HIDE_SPRITES;
}

void menu()
{
  UBYTE i, key;
  UBYTE slowdown;
  UBYTE cursor;

  slowdown = 50;

  for(i = 0; i < NB_TOOLS; i++)
    set_icon(i,
	     i == FIRST_TOOL + current_tool ||
	     i == FIRST_COLOR + current_color ||
	     i == FIRST_MODE + current_mode);

  cursor = current_cursor;
  current_cursor = ARROW;
  set_cursor();
  move_menu_cursor();
  SHOW_SPRITES;
  waitpadup();
  do {
    wait_vbl_done();
    key = joypad();
    if(key & (J_UP|J_DOWN|J_LEFT|J_RIGHT)) {
      if(key & J_UP)
	menu_cursor_pos = icons[menu_cursor_pos].up;
      if(key & J_DOWN)
	menu_cursor_pos = icons[menu_cursor_pos].down;
      if(key & J_LEFT)
	menu_cursor_pos = icons[menu_cursor_pos].left;
      if(key & J_RIGHT)
	menu_cursor_pos = icons[menu_cursor_pos].right;
      move_menu_cursor();
      while(slowdown && key == joypad()) {
	wait_vbl_done();
	slowdown--;
      }
      slowdown = 10;
    } else
      slowdown = 50;
    if(key & J_A) {
      if( /* menu_cursor_pos >= FIRST_TOOL && */ menu_cursor_pos <= LAST_TOOL) {
	if(menu_cursor_pos != /* FIRST_TOOL + */ current_tool) {
	  change_icon(/* FIRST_TOOL + */ current_tool, UNSELECTED);
	  current_tool = menu_cursor_pos /* - FIRST_TOOL */;
	  change_icon(/* FIRST_TOOL + */ current_tool, SELECTED);
	  cursor = icons[/* FIRST_TOOL + */ current_tool].cursor;
	}
      } else if(menu_cursor_pos >= FIRST_COLOR && menu_cursor_pos <= LAST_COLOR) {
	if(menu_cursor_pos != FIRST_COLOR + current_color) {
	  change_icon(FIRST_COLOR + current_color, UNSELECTED);
	  current_color = menu_cursor_pos - FIRST_COLOR;
	  change_icon(FIRST_COLOR + current_color, SELECTED);
	}
      } else if(menu_cursor_pos >= FIRST_MODE && menu_cursor_pos <= LAST_MODE) {
	if(menu_cursor_pos != FIRST_MODE + current_mode) {
	  change_icon(FIRST_MODE + current_mode, UNSELECTED);
	  current_mode = menu_cursor_pos - FIRST_MODE;
	  change_icon(FIRST_MODE + current_mode, SELECTED);
	}
      }
    }
  } while(key != J_SELECT);
  waitpadup();
  for(i = 0; i < NB_TOOLS; i++)
    reset_icon(i);
  HIDE_SPRITES;
  current_cursor = cursor;
}

void run()
{
  UBYTE key;
  UBYTE slowdown;
  UBYTE drawn, erased;

  slowdown = 10;
  drawn = erased = 0;

  set_cursor();
  move_cursor();
  SHOW_SPRITES;

  while(1) {
    wait_vbl_done();
    key = joypad();
    if(key & (J_UP|J_DOWN|J_LEFT|J_RIGHT)) {
      if(key & J_UP && cursor_y > 0)
	cursor_y--;
      if(key & J_DOWN && cursor_y < 143)
	cursor_y++;
      if(key & J_LEFT && cursor_x > 0)
	cursor_x--;
      if(key & J_RIGHT && cursor_x < 159)
	cursor_x++;
      move_cursor();
      while(slowdown && key == joypad()) {
	wait_vbl_done();
	slowdown--;
      }
      slowdown = 1;
      drawn = erased = 0;
    } else
      slowdown = 10;
    if(key & J_SELECT) {
      HIDE_SPRITES;
      menu();
      set_cursor();
      move_cursor();
      SHOW_SPRITES;
    }
    if((key & (J_A|J_B)) == J_A) {
      if(!drawn) {
	drawn++;
	plot(cursor_x, cursor_y, colors[current_color], modes[current_mode]);
      }
    } else
      drawn = 0;
    if((key & (J_A|J_B)) == J_B) {
      if(!erased) {
	erased++;
	plot(cursor_x, cursor_y, WHITE, SOLID);
      }
    } else
      erased = 0;
  }
}

void main()
{
  /* Initialize sprite palette */
  OBP1_REG = 0xE0U;

  splash();

  current_tool = 0;
  current_color = BLACK;
  current_mode = SOLID;
  current_cursor = PEN;
  menu_cursor_pos = 0;
  cursor_x = 160/2;
  cursor_y = 144/2;

  run();
}
