/*								-*- C++ -*-
 * $Id: GDI_bitmap.cpp,v 1.2 1998/08/24 22:45:51 wg Exp $
 *
 * Purpose: bitmap classes to implement pixmaps, icons, and cursors
 *
 * Authors: Markus Holzem and Julian Smart
 *
 * Copyright: (C) 1995, AIAI, University of Edinburgh (Julian)
 * Copyright: (C) 1995, GNU (Markus)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Additionally everyone using this library has to announce it with:
 *
 *   This software uses the wxWindows-Xt GUI library
 *   (C) Markus Holzem, available via
 *       ftp://ftp.aiai.ed.ac.uk/pub/packages/wxwin/ports/xt
 */

#ifdef __GNUG__
#pragma implementation "GDI_bitmap.h"
#endif

#define  Uses_XLib
#define  Uses_wxBitmap
#define  Uses_wxColour
#include "wx.h"

#if USE_XPM
    #include XPM_INCLUDE
#endif

#include <X11/cursorfont.h>
#include <default.xbm>

// hints for what to free in wxBitmap::Destroy()
enum {
    __BITMAP_NORMAL,	// <-- no special data
#if USE_XPM
    __BITMAP_XPM,	// <-- XpmAttributes
#endif
};

IMPLEMENT_DYNAMIC_CLASS(wxBitmap, wxObject)
IMPLEMENT_DYNAMIC_CLASS(wxCursor, wxBitmap)
IMPLEMENT_DYNAMIC_CLASS(wxIcon, wxBitmap)
IMPLEMENT_DYNAMIC_CLASS(wxGDIList, wxList)

//-----------------------------------------------------------------------------
// wxBitmapRep
//-----------------------------------------------------------------------------

class wxBitmapRep : public wxObject {
DECLARE_DYNAMIC_CLASS(wxBitmapRep)
public:
    int          type;			// what is the type of the bitmap
    unsigned int width, height, depth;	// dimensions of bitmap
    int          x_hot, y_hot;		// hotspot of bitmap
    Pixmap       x_pixmap;		// the displayable pixmap
    Pixmap       x_mask;		// mask, which pixels of x_pixmap are drawn
    Cursor       x_cursor;
    wxColourMap* cmap;
    int          ref_count;
    // Optional stuff for different bitmaps
    union {
#if USE_XPM
	XpmAttributes* xpm;		// for XPM pixmaps
#endif
	int dummy;			// for savety
    } ext;
#if WXDEBUG
    void Dump(ostream& str);
#endif
};

IMPLEMENT_ABSTRACT_CLASS(wxBitmapRep, wxObject);

#if WXDEBUG
void wxBitmapRep::Dump(ostream& str)
{
    str << "wxBitmapRep ref = " << ref_count;
}
#endif // WXDEBUG

//-----------------------------------------------------------------------------
// wxBitmap
//-----------------------------------------------------------------------------

wxBitmap::wxBitmap(void)
{
    __type = wxTYPE_BITMAP;

    bmp = NULL;

#if !WXGARBAGE_COLLECTION_ON
    wxTheBitmapList->Append(this);
#endif
}

wxBitmap::wxBitmap(wxBitmap& new_bmp)
{
    __type = wxTYPE_BITMAP;

    bmp = new_bmp.bmp;
    if (bmp) bmp->ref_count++;

#if !WXGARBAGE_COLLECTION_ON
    wxTheBitmapList->Append(this);
#endif
}

wxBitmap::wxBitmap(char bits[], int w, int h, int d)
{
    wxASSERT(d==1, "wxBitmap(bits,w,h,d): cannot create bitmap with depth != 1");

    __type = wxTYPE_BITMAP;

    bmp = wxNEW wxBitmapRep;

    bmp->type      = __BITMAP_NORMAL;
    bmp->width     = w;
    bmp->height    = h;
    bmp->depth     = d;
    bmp->x_hot     = bmp->y_hot = 0;
    bmp->x_pixmap  = None;
    bmp->x_mask    = None;
    bmp->x_cursor  = 0;
    bmp->cmap      = wxAPP_COLOURMAP;
    bmp->ref_count = 1;
    // create pixmap with depth 1
    if (((bmp->x_pixmap
	  = XCreateBitmapFromData(wxAPP_DISPLAY, wxAPP_ROOT, bits, w, h)))
	== None)
    {
	// create failed
	delete bmp; bmp = NULL;
    }

#if !WXGARBAGE_COLLECTION_ON
    wxTheBitmapList->Append(this);
#endif
}

wxBitmap::wxBitmap(char *bitmap_file, long flags)
{
    __type = wxTYPE_BITMAP;

    bmp = NULL;
    // use load method
    (void)LoadFile(bitmap_file, flags);

#if !WXGARBAGE_COLLECTION_ON
    wxTheBitmapList->Append(this);
#endif
}

#if USE_XPM

wxBitmap::wxBitmap(char **data, wxItem *WXUNUSED(anItem)) // anItem used for MOTIF
{
    __type = wxTYPE_BITMAP;

    bmp = wxNEW wxBitmapRep;

    bmp->x_cursor  = 0;
    bmp->cmap      = wxAPP_COLOURMAP;
    bmp->ref_count = 1;
    // what I want to get from XPM
    bmp->ext.xpm = wxNEW XpmAttributes;
    bmp->ext.xpm->valuemask = XpmReturnInfos | XpmReturnPixels | XpmCloseness;
    bmp->ext.xpm->closeness = 40000;
    // create pixmap
    int ErrorStatus
	= XpmCreatePixmapFromData(wxAPP_DISPLAY, wxAPP_ROOT,
				  data, &(bmp->x_pixmap),
				  &(bmp->x_mask), bmp->ext.xpm);
    if (ErrorStatus == XpmSuccess) {
	// create pixmap successful
	bmp->type   = __BITMAP_XPM;
	bmp->width  = bmp->ext.xpm->width;
	bmp->height = bmp->ext.xpm->height;
	bmp->x_hot  = bmp->ext.xpm->x_hotspot;
	bmp->y_hot  = bmp->ext.xpm->y_hotspot;
	// get depth of pixmap
	int sdummy; unsigned int udummy; Window wdummy;
	XGetGeometry(wxAPP_DISPLAY, bmp->x_pixmap, &wdummy, &sdummy, &sdummy,
		     &udummy, &udummy, &udummy, &(bmp->depth));
    } else {
	// create failed: free all memory
	XpmFreeAttributes(bmp->ext.xpm); delete bmp->ext.xpm;
	delete bmp; bmp = NULL;
    }

#if !WXGARBAGE_COLLECTION_ON
    wxTheBitmapList->Append(this);
#endif
}

#endif

wxBitmap::wxBitmap(int w, int h, int d)
{
    __type = wxTYPE_BITMAP;

    bmp = NULL;
    // use create method
    (void)Create(w, h, d);

#if !WXGARBAGE_COLLECTION_ON
    wxTheBitmapList->Append(this);
#endif
}

wxBitmap::~wxBitmap(void)
{
    // free pixmap and infos
    Destroy();

#if !WXGARBAGE_COLLECTION_ON
    // remove Bitmap form List
    wxTheBitmapList->DeleteObject(this);
#endif
}

//-----------------------------------------------------------------------------
// assignment operators to handle reference counting
//-----------------------------------------------------------------------------

wxBitmap& wxBitmap::operator = (wxBitmap& new_bmp)
{
    Destroy(); // unreference old bitmap
    bmp = new_bmp.bmp;
    if (bmp) bmp->ref_count++;
    // return this for further assignments
    return (*this);
}

void wxBitmap::operator = (wxBitmap* p_new_bmp)
{
    Destroy(); // unreference old bitmap
    // new_bmp has to ok! otherwise THIS bitmap remains empty
    if (p_new_bmp) {
	bmp = p_new_bmp->bmp;
	if (bmp) bmp->ref_count++;
    }
}

//-----------------------------------------------------------------------------
// create and destroy bitmap
//-----------------------------------------------------------------------------

Bool wxBitmap::Create(int w, int h, int d)
{
    Destroy(); // destroy old bitmap if any

    bmp = wxNEW wxBitmapRep;

    // set pixmap specific data
    bmp->type      = __BITMAP_NORMAL;
    bmp->width     = w;
    bmp->height    = h;
    bmp->depth     = d < 1 ? wxDisplayDepth() : d;
    bmp->x_hot     = 0;
    bmp->y_hot     = 0;
    bmp->x_cursor  = 0;
    bmp->cmap      = wxAPP_COLOURMAP;
    bmp->ref_count = 1;
    bmp->x_mask    = None;
    // create pixmap
    if (((bmp->x_pixmap
	  = XCreatePixmap(wxAPP_DISPLAY, wxAPP_ROOT, w, h, bmp->depth)))
	== None)
    {
	// create failed!
	delete bmp; bmp = NULL;
    }
    return Ok();
}

void wxBitmap::Destroy(void)
{
    // unreference to old bitmap and destroy it if necessary
    if (bmp && --bmp->ref_count==0) {
	XFreePixmap(wxAPP_DISPLAY, bmp->x_pixmap); // free pixmap
	if (bmp->x_mask != None)
	    XFreePixmap(wxAPP_DISPLAY, bmp->x_mask); // free mask
	switch (bmp->type) { // free type specific data
#if USE_XPM
	case __BITMAP_XPM:
	    // free XPM data
	    XFreeColors(wxAPP_DISPLAY, bmp->cmap->GetColormap(),
			bmp->ext.xpm->pixels, bmp->ext.xpm->npixels, 0);
	    XpmFreeAttributes(bmp->ext.xpm);
	    delete bmp->ext.xpm;
	    break;
#endif
	default:
	    break; // no other formats so far
	}
	delete bmp;
    }
    // contains no pixmap
    bmp = NULL;
}

//-----------------------------------------------------------------------------
// load and save bitmaps
//-----------------------------------------------------------------------------

Bool wxBitmap::LoadFile(char *fname, long flags)
{
    Destroy(); // destroy old pixmap if any

    if (!wxFileExists(fname)) { // if file does not exist, stop processing
	return FALSE;
    }

    bmp = wxNEW wxBitmapRep;

    bmp->x_cursor  = 0;
    bmp->cmap      = wxAPP_COLOURMAP;
    bmp->ref_count = 1;
    bmp->x_mask    = None;

    if (flags & wxBITMAP_TYPE_XBM) { // XBM file format
	if (XReadBitmapFile(wxAPP_DISPLAY, wxAPP_ROOT, fname,
			    &(bmp->width), &(bmp->height),
			    &(bmp->x_pixmap), 
			    &(bmp->x_hot), &(bmp->y_hot))
	    == BitmapSuccess)
	{
	    bmp->type  = __BITMAP_NORMAL;
	    bmp->depth = 1;
	} else {
	    delete bmp;
	    bmp = NULL;
	}
    }
#if USE_XPM
    else if (flags & wxBITMAP_TYPE_XPM) { // XPM file format
	// what I want to get
	bmp->ext.xpm = wxNEW XpmAttributes;
	bmp->ext.xpm->valuemask = XpmReturnInfos | XpmReturnPixels | XpmCloseness;
	bmp->ext.xpm->closeness = 40000;

	if (XpmReadFileToPixmap(wxAPP_DISPLAY, wxAPP_ROOT, fname,
				&(bmp->x_pixmap), &(bmp->x_mask), bmp->ext.xpm)
	    == XpmSuccess)
	{
	    // read pixmap ok!
	    bmp->type   = __BITMAP_XPM;
	    bmp->width  = bmp->ext.xpm->width;
	    bmp->height = bmp->ext.xpm->height;
	    bmp->x_hot  = bmp->ext.xpm->x_hotspot;
	    bmp->y_hot  = bmp->ext.xpm->y_hotspot;
	    int sdummy; unsigned int udummy; Window wdummy;
	    XGetGeometry(wxAPP_DISPLAY, bmp->x_pixmap, &wdummy, &sdummy, &sdummy,
			 &udummy, &udummy, &udummy, &(bmp->depth));
	} else {
	    // read failed: free all memory
	    XpmFreeAttributes(bmp->ext.xpm);
	    delete bmp->ext.xpm;
	    delete bmp;
	    bmp = NULL;
	}
    }
#endif
#if USE_IMAGE_LOADING
    else if (flags & (wxBITMAP_TYPE_ANY | wxBITMAP_TYPE_BMP | wxBITMAP_TYPE_GIF)) {
	// use wxImage for loading
#endif
    else {
	// I don't know what to do with this type
	delete bmp;
	bmp = NULL;
    }
    return Ok();
}

Bool wxBitmap::SaveFile(char *fname, int type, wxColourMap *WXUNUSED(cmap))
{
    if (bmp) {
	switch (type) {
	case wxBITMAP_TYPE_XBM:
	    if (bmp->depth == 1)
		return (XWriteBitmapFile(wxAPP_DISPLAY, fname, bmp->x_pixmap,
					 bmp->width, bmp->height,
					 bmp->x_hot, bmp->y_hot)
			== BitmapSuccess);
	    break; // write failed or depth != 1
#if USE_XPM
	case wxBITMAP_TYPE_XPM:
	    return (XpmWriteFileFromPixmap(wxAPP_DISPLAY, fname, bmp->x_pixmap,
					   (bmp->depth == 1 ? bmp->x_pixmap : bmp->x_mask),
					   (XpmAttributes*)NULL)
		    == XpmSuccess);
	    break; // write failed
#endif
	default:
	    break; // no other save methods so far
	}
    }
    return FALSE;
}

//-----------------------------------------------------------------------------
// get attributes of bitmap
//-----------------------------------------------------------------------------

int wxBitmap::GetDepth(void)
{
    return (bmp ? bmp->depth : 1);
}

int wxBitmap::GetHeight(void)
{
    return (bmp ? bmp->height : 0);
}

int wxBitmap::GetWidth(void)
{
    return (bmp ? bmp->width : 0);
}

void  wxBitmap::GetHotSpot(int *x, int *y)
{
    if (bmp) { *x = bmp->x_hot; *y = bmp->y_hot; }
    else     { *x = *y = 0; }
}

wxColourMap* wxBitmap::GetColourMap(void)
{
    return (bmp ? bmp->cmap : wxAPP_COLOURMAP);
}

void wxBitmap::SetColourMap(wxColourMap *new_cmap)
{
    bmp->cmap = (new_cmap ? new_cmap : wxAPP_COLOURMAP);
}

//-----------------------------------------------------------------------------
// Get X representation
//-----------------------------------------------------------------------------

Bool wxBitmap::Ok(void)
{
    return (bmp != NULL);
}

Pixmap wxBitmap::GetPixmap(void) // for wxBitmap and wxIcon
{
    if (!bmp || bmp->x_pixmap==None) {
	wxError("no valid pixmap - using fallback pixmap");
	return (wxAPP_FALLBACK_PIXMAP);
    }
    return (bmp->x_pixmap);
}

Pixmap wxBitmap::GetMask(void) // for wxBitmap and wxIcon
{
    if (!bmp || bmp->x_pixmap==None) {
	wxError("no valid pixmap - using fallback pixmap");
	return (wxAPP_FALLBACK_PIXMAP);
    }
    if (bmp->depth == 1)
	return (bmp->x_pixmap);
    return (bmp->x_mask);
}

//-----------------------------------------------------------------------------
// wxCursor
//-----------------------------------------------------------------------------

static XColor black = { 0, 0, 0, 0, 0, 0 };

wxCursor::wxCursor(void) : wxBitmap()
{
    __type = wxTYPE_CURSOR;
}

wxCursor::wxCursor(wxCursor& new_cursor) : wxBitmap()
{
    __type = wxTYPE_CURSOR;

    bmp = new_cursor.bmp;
    if (bmp) bmp->ref_count++;
}

wxCursor::wxCursor(char *name, long flags) : wxBitmap(name, flags)
{
    __type = wxTYPE_CURSOR;

    if (bmp && bmp->depth == 1) { // only possible for bitmaps
	bmp->x_cursor
	    = XCreatePixmapCursor(wxAPP_DISPLAY,
				  bmp->x_pixmap, bmp->x_pixmap,
				  &black, &black,
				  bmp->x_hot, bmp->y_hot);
	XFreePixmap(wxAPP_DISPLAY, bmp->x_pixmap); // free pixmap
    }
}

wxCursor::wxCursor(char bits[], int width, int height, int x, int y,
		   char *maskbits)
    : wxBitmap(bits, width, height, 1)
{
    __type = wxTYPE_CURSOR;

    if (bmp) {
	bmp->x_hot = (x == -1) ? width/2 : x;
	bmp->y_hot = (y == -1) ? height/2 : y;
	Pixmap mask_pixmap = maskbits ?
	    XCreateBitmapFromData(wxAPP_DISPLAY, wxAPP_ROOT, maskbits,
				  width, height) :
	    bmp->x_pixmap;
	XColor bg;
	bg.pixel = WhitePixelOfScreen(wxAPP_SCREEN);
	XQueryColor(wxAPP_DISPLAY, wxAPP_COLOURMAP->GetColormap(), &bg);
	bmp->x_cursor
	    = XCreatePixmapCursor(wxAPP_DISPLAY,
				  bmp->x_pixmap, mask_pixmap,
				  &black, &bg,
				  bmp->x_hot, bmp->y_hot);
	if (maskbits)
	    XFreePixmap(wxAPP_DISPLAY, mask_pixmap);
	XFreePixmap(wxAPP_DISPLAY, bmp->x_pixmap); // free pixmap
    }
}

wxCursor::wxCursor(int cursor_type) : wxBitmap()
{
    __type = wxTYPE_CURSOR;

    static const unsigned int x_cursor_id[] = {
	XC_top_left_arrow,    // wxCURSOR_ARROW		
	XC_based_arrow_down,  // wxCURSOR_BASED_ARROW_DOWN
	XC_based_arrow_up,    // wxCURSOR_BASED_ARROW_UP
	XC_target,            // wxCURSOR_BULLSEYE
	XC_crosshair,         // wxCURSOR_CROSS
	XC_cross_reverse,     // wxCURSOR_CROSS_REVERSE
	XC_double_arrow,      // wxCURSOR_DOUBLE_ARROW
	XC_hand1,             // wxCURSOR_HAND
	XC_xterm,             // wxCURSOR_IBEAM
	XC_leftbutton,        // wxCURSOR_LEFT_BUTTON
	XC_sizing,            // wxCURSOR_MAGNIFIER
	XC_middlebutton,      // wxCURSOR_MIDDLE_BUTTON
	XC_fleur,             // wxCURSOR_MOVE
	XC_pirate,            // wxCURSOR_NO_ENTRY
	XC_spraycan,          // wxCURSOR_PAINT_BRUSH
	XC_pencil,            // wxCURSOR_PENCIL
	XC_sb_left_arrow,     // wxCURSOR_POINT_LEFT
	XC_sb_right_arrow,    // wxCURSOR_POINT_RIGHT
	XC_question_arrow,    // wxCURSOR_QUESTION_ARROW
	XC_rightbutton,       // wxCURSOR_RIGHT_BUTTON
	XC_circle,            // wxCURSOR_SIZENESW
	XC_sb_v_double_arrow, // wxCURSOR_SIZENS
	XC_circle,            // wxCURSOR_SIZENWSE
	XC_sb_h_double_arrow, // wxCURSOR_SIZEWE
	XC_sizing,            // wxCURSOR_SIZING
	XC_spraycan,          // wxCURSOR_SPRAYCAN
	XC_watch,             // wxCURSOR_WAIT
	XC_watch,             // wxCURSOR_WATCH
    };

    static char char_data[32] = { // bits for char pointer
	0xFF, 0x00, 0xFF, 0x00, 0xFF, 0x00, 0xFF, 0x00,
	0xFF, 0x00, 0xFF, 0x00, 0xFF, 0x00, 0xFF, 0x00,
	0xFF, 0x00, 0xFF, 0x00, 0xFF, 0x00, 0xFF, 0x00,
	0xFF, 0x00, 0xFF, 0x00, 0xFF, 0x00, 0xFF, 0x00,
    };

    static char blank_data[32] = { // bits for blank pointer
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    __type = wxTYPE_CURSOR;

    bmp = wxNEW wxBitmapRep;

    bmp->type      = __BITMAP_NORMAL;
    bmp->width     = 16; // size of 16x16 is a good bet. blank and char cursor
    bmp->height    = 16; // have this size and for font_cursors the size cannot be queried
    bmp->depth     = 1;
    bmp->x_hot     = bmp->y_hot = 0;
    bmp->x_pixmap  = 0;  // remains unset even for blank and char cursor
    bmp->x_cursor  = 0;
    bmp->cmap      = wxAPP_COLOURMAP;
    bmp->ref_count = 1;

    Pixmap pixmap;
    
    switch (cursor_type) {
    case wxCURSOR_BLANK:
	pixmap = XCreateBitmapFromData(wxAPP_DISPLAY, wxAPP_ROOT, blank_data, 16, 16);
	bmp->x_cursor
	    = XCreatePixmapCursor(wxAPP_DISPLAY, pixmap, pixmap, &black, &black, 8, 8);
	bmp->x_hot = bmp->y_hot = 8;
	XFreePixmap(wxAPP_DISPLAY, pixmap);
	break;
    case wxCURSOR_CHAR:
	pixmap = XCreateBitmapFromData(wxAPP_DISPLAY, wxAPP_ROOT, char_data, 16, 16);
	bmp->x_cursor
	    = XCreatePixmapCursor(wxAPP_DISPLAY, pixmap, pixmap, &black, &black, 0, 13);
	bmp->x_hot = 0; bmp->y_hot = 13;
	XFreePixmap(wxAPP_DISPLAY, pixmap);
	break;
    default:
	if (wxFIRST_X11_CURSOR <= cursor_type && cursor_type <= wxLAST_X11_CURSOR)
	    bmp->x_cursor
		= XCreateFontCursor(wxAPP_DISPLAY, x_cursor_id[cursor_type]);
	break;
    }
    if (!bmp->x_cursor) {
	// create failed
	delete bmp; bmp = NULL;
    }
}

wxCursor::~wxCursor(void)
{
    if (bmp && bmp->ref_count == 1) {
	// free cursor on when only one reference is left
	XFreeCursor(wxAPP_DISPLAY, bmp->x_cursor);
	// delete representation => no XFreePixmap (cursors don't have pixmaps)
	delete bmp;
	bmp = NULL;
    }
    // else: ~wxBitmap counts references down
}

//-----------------------------------------------------------------------------
// change foreground and background colour
//-----------------------------------------------------------------------------

void wxCursor::ChangeColour(wxColour* fg, wxColour *bg)
{
    if (!bmp || !bmp->x_cursor || !fg || !bg)
	return; // I need at least a cursor and the colours

    // query colours
    XColor fore, back; Colormap cmap = wxAPP_COLOURMAP->GetColormap();
    fore.pixel = fg->GetPixel(); XQueryColor(wxAPP_DISPLAY, cmap, &fore);
    back.pixel = bg->GetPixel(); XQueryColor(wxAPP_DISPLAY, cmap, &back);
    // change colour of cursor
    XRecolorCursor(wxAPP_DISPLAY, bmp->x_cursor, &fore, &back);
}

//-----------------------------------------------------------------------------
// Get X representation
//-----------------------------------------------------------------------------

Cursor wxCursor::GetCursor(void) // for wxCursor
{
    if (!bmp || !bmp->x_cursor) {
	wxError("no valid cursor - using fallback cursor");
	return (wxAPP_FALLBACK_CURSOR);
    }
    return (bmp->x_cursor);
}

//-----------------------------------------------------------------------------
// wxIcon
//-----------------------------------------------------------------------------

wxIcon::wxIcon(void) : wxBitmap()
{
    __type = wxTYPE_ICON;
}

wxIcon::wxIcon(wxIcon& new_icon) : wxBitmap()
{
    __type = wxTYPE_ICON;

    bmp = new_icon.bmp;
    if (bmp) bmp->ref_count++;
}

wxIcon::wxIcon(char bits[], int width, int height, int depth)
    : wxBitmap(bits, width, height, depth)
{
    __type = wxTYPE_ICON;
}

wxIcon::wxIcon(int width, int height, int depth)
    : wxBitmap(width, height, depth)
{
    __type = wxTYPE_ICON;
}

#if USE_XPM
wxIcon::wxIcon(char **data)
    : wxBitmap(data)
{
    __type = wxTYPE_ICON;
}
#endif

wxIcon::wxIcon(char *name, long flags)
    : wxBitmap(name, flags)
{
    __type = wxTYPE_ICON;
}

//-----------------------------------------------------------------------------
// assignment operators to handle reference counting
//-----------------------------------------------------------------------------

wxIcon& wxIcon::operator = (wxIcon& new_icon)
{
    Destroy(); // unreference old bitmap
    bmp = new_icon.bmp;
    if (bmp) bmp->ref_count++;
    // return this for further assignments
    return (*this);
}

void wxIcon::operator = (wxIcon* p_new_icon)
{
    Destroy(); // unreference old bitmap
    // new_bmp has to ok! otherwise THIS bitmap remains empty
    if (p_new_icon) {
	bmp = p_new_icon->bmp;
	if (bmp) bmp->ref_count++;
    }
}

//-----------------------------------------------------------------------------
// GDIList
//-----------------------------------------------------------------------------

wxGDIList::wxGDIList(void) : wxList()
{
    #include "./DLG_error.xbm"
    #include "./DLG_info.xbm"
    #include "./DLG_noenter.xbm"
    #include "./DLG_question.xbm"
    #include "./DLG_warning.xbm"

    // create a standard cursor and a standard pixmap for the application
    // to be used when cursor or pixmap creation failed
    wxAPP_FALLBACK_CURSOR = XCreateFontCursor(wxAPP_DISPLAY, XC_top_left_arrow);
    wxAPP_FALLBACK_PIXMAP = XCreateBitmapFromData(wxAPP_DISPLAY, wxAPP_ROOT,
						  (char*)default_bits, default_width,
						  default_height);
    if (!wxAPP_FALLBACK_CURSOR || !wxAPP_FALLBACK_PIXMAP)
	wxError("could not create fallback cursor or fallback pixmap - BEWARE");

    // create standard cursors
    // there is one representation of wxGDIList
    wxTheBitmapList = this;
    // standard preloaded bitmaps, icons and cursors
    wxSTANDARD_CURSOR  = wxNEW wxCursor(wxCURSOR_ARROW);
    wxHOURGLASS_CURSOR = wxNEW wxCursor(wxCURSOR_WAIT);
    wxCROSS_CURSOR     = wxNEW wxCursor(wxCURSOR_CROSS);
    wxMOVE_CURSOR      = wxNEW wxCursor(wxCURSOR_MOVE);

    wxDLG_ERROR_ICON    = wxNEW wxIcon(err_bits, err_wd, err_ht, 1);
    wxDLG_INFO_ICON     = wxNEW wxIcon(info_bits, info_wd, info_ht, 1);
    wxDLG_NOENTER_ICON  = wxNEW wxIcon(noenter_bits, noenter_wd, noenter_ht, 1);
    wxDLG_QUESTION_ICON = wxNEW wxIcon(question_bits, question_wd, question_ht, 1);
    wxDLG_WARNING_ICON  = wxNEW wxIcon(warning_bits, warning_wd, warning_ht, 1);
}

wxGDIList::~wxGDIList (void)
{
    // free callback cursor and pixmap
    XFreeCursor(wxAPP_DISPLAY, wxAPP_FALLBACK_CURSOR);
    XFreePixmap(wxAPP_DISPLAY, wxAPP_FALLBACK_PIXMAP);
    // delete bitmaps, icons, and cursors allocated by the application
/*  FUM-Warning PURIFY
    wxNode *node = First();
    while (node) {
	wxBitmap *bitmap = (wxBitmap*)node->Data();
	wxNode *next = node->Next();
	delete bitmap;
	node = next;
    }
*/
}
