/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    5635-34 Springhouse Dr.
    Pleasanton, CA 94588 (USA)
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_timer.c,v 1.4 1999/07/21 06:08:27 slouken Exp $";
#endif

#include <be/kernel/OS.h>

#include "SDL_thread.h"
#include "SDL_timer.h"

#define ROUND_RESOLUTION(X)	\
	(((X+TIMER_RESOLUTION-1)/TIMER_RESOLUTION)*TIMER_RESOLUTION)

static bigtime_t start;

void SDL_StartTicks(void)
{
	/* Set first ticks value */
	start = system_time();
}

Uint32 SDL_GetTicks(void)
{
	return((system_time()-start)/1000);
}

void SDL_Delay(Uint32 ms)
{
	snooze(ms*1000);
}

/* Data to handle a single periodic alarm */
static int timer_alive = 0;
static SDL_TimerCallback alarm_callback = NULL;
static Uint32 alarm_interval;
static Uint32 alarm_pending;
static SDL_Thread *timer = NULL;

static int RunTimer(void *unused)
{
	Uint32 ms;

	while ( timer_alive ) {
		SDL_Delay(10);
		if ( alarm_pending ) {
			--alarm_pending;
			if ( ! alarm_pending ) {
				ms = (*alarm_callback)(alarm_interval);
				if ( ms != alarm_interval ) {
					alarm_interval = ROUND_RESOLUTION(ms);
				}
				alarm_pending = (alarm_interval/10);
			}
		}
	}
	return(0);
}

int SDL_SetTimer(Uint32 ms, SDL_TimerCallback callback)
{
	alarm_callback = callback;
	alarm_interval = ROUND_RESOLUTION(ms);
	alarm_pending = (alarm_interval/10);
	return(0);
}

int SDL_TimerInit(void)
{
	timer_alive = 1;
	timer = SDL_CreateThread(RunTimer, NULL);
	if ( timer == NULL )
		return(-1);
	return(0);
}

void SDL_TimerQuit(void)
{
	timer_alive = 0;
	if ( timer ) {
		SDL_WaitThread(timer, NULL);
		timer = NULL;
	}
}
