// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: notify.h,v 1.3 1998/09/12 02:46:37 jgg Exp $
/* ######################################################################

   Notify - Notification mechanism
   Notifyer List - List of notifications
   Function Notifyer - Notification to call a function on trigger
   
   This class is provided by the widget library to provide client level
   notification of events generated or recived by the widget.   
   This allows the separation of Client from Class Library and simplifies
   the creation of re-usable notification handlers.
   
   The base widget class has a Notify list which stores a linked list of
   Notifyer classes. Each notifyer has a unique ID (Type+SubType). When the
   widget wishes to generate a notification it requests the NotifyerList
   to Trigger all the Notifyers that are regiesterd for that ID. The
   result is reduced to a single bool that is the logical and of all
   the returned bool's of the trigger function. The default if there
   is no notifications it to return true.

   The NotifyFunction handler is a simple Notifyer to call a user specified
   function on Trigger. The convience function AttachCallback will attach
   the given function to the given notification on a given widget.

   Type ID's are a static member inside of each widget class, 
   <Widget>::Type where <widget> is the name of the widget generating the
   notification. To hide this from the end user, who doesn't care which 
   widget class owns the event each widget has #defines like:
      #define Nt_<thing> Notifyer::Tag(<Widget>::Type,0)
   Thing is the notifyer type ie, Nt_Resize.
   Widget is the owning widget ie Widget
   0 is an aribitary integer that is unique for the widget. 
 
   Unique Widget::Type values are generated by the system linker. Each
   class has a 
      static void *Type;;
    Line and an associated
      void *<Widget>::Type = (void *)(&<Widget>::Type);
   Which will generate a unique number.

   Notifications can be realized to class members with the ClassNotifyer
   class. Note that V-Funcs likely will not work.. The syntax is:
        new ClassNotifyer<Class>(this,Nt_Something,CBFunc);
   
   ##################################################################### */
									/*}}}*/
// Header section: deity
#ifndef DEITY_NOTIFY_H
#define DEITY_NOTIFY_H

#ifdef __GNUG__
#pragma interface "deity/notify.h"
#endif  

class Widget;
class Notifyer;
class NotifyerList;

// Base notifyer
class Notifyer
{
   friend NotifyerList;
   
   // Next pointer in the linked list
   Notifyer *Next;

   public:

   // Notification Tag
   struct Tag
   {
      void *Type;
      unsigned long SubType;
      
      inline bool operator ==(const Tag &rhs) const {return Type == rhs.Type && SubType == rhs.SubType;};
      inline Tag(void *Type,unsigned long SubType) : Type(Type),
          SubType(SubType) {};
      Tag() : Type(0), SubType(0) {};
   };
   
   protected:

   // Notification Type
   Tag ID;
   
   public:
   
   // Trigger function
   virtual bool Trigger(Widget *From,Tag ID,void *Data) = 0;

   // Construct/Destruct
   Notifyer(Tag Id);
   virtual ~Notifyer() {};
};

// List of notifications
class NotifyerList
{
   // Head of notification linked list
   Notifyer *List;
   
   public:

   // Manipulators
   void Add(Notifyer *Notif);
   bool Trigger(Widget *From,Notifyer::Tag ID,void *Data);
   
   // Construct/destruct
   NotifyerList();
   ~NotifyerList();
};

class FuncNotifyer : public Notifyer
{
   public:
   typedef bool (*TFunc)(Widget *From, Tag Id, void *Data);

   protected:
   TFunc Func;
   
   public:
   
   virtual bool Trigger(Widget *From,Tag Id,void *Data) {return Func(From,Id,Data);};
   
   FuncNotifyer(Tag Id,TFunc Func);
};

template <class T> class ClassNotifyer : public Notifyer
{
   public:
   typedef bool (T::* TFunc)(Widget *From, Tag Id, void *Data);
   
   protected:
   T *Cls;
   TFunc Func;
   
   public:
   
   virtual bool Trigger(Widget *From,Tag Id,void *Data) {return (Cls->*Func)(From,Id,Data);};
   ClassNotifyer(T *Class,Tag Id,TFunc Func) : Notifyer(Id), Cls(Class), Func(Func) {};
};

void AttachCallback(Widget *To,Notifyer::Tag Id,FuncNotifyer::TFunc Func);

#endif
