#|------------------------------------------------------------*-Scheme-*--|
 | File:    modules/corelib/safeglue.scm
 |
 |          Copyright (C)1997 Donovan Kolbly <d.kolbly@rscheme.org>
 |          as part of the RScheme project, licensed for free use.
 |          See <http://www.rscheme.org/> for the latest information.
 |
 | File version:     1.9
 | File mod date:    1997.11.29 23:10:38
 | System build:     v0.7.2, 97.12.21
 |
 | Purpose:          Provide the 'define-safe-glue' rewriter
 |------------------------------------------------------------------------|
 | Notes:
 |      This rewriter makes writing C glue functions safter, because
 |      you can use usual RScheme syntax for defining the number and
 |      types of the arguments, and this rewriter will install the
 |      checks (and conversion code) for you.
 |      
 |      Also note:
 |      (1) This was moved here from the unixm module to share code
 |      among modules, and to provide the safe-glue functionality
 |      to the little bit of glue present in the core system
 |      
 |      (2) This is the latest version which supports complex (x11-module)
 |      types and named "views"
 `------------------------------------------------------------------------|#

#|
	       The "Safe Glue" Wrapper for define-glue
	       =======================================

   define-safe-glue is a rewriter-macro wrapper around define-glue
   that constructs argument checking and massaging code.  This
   is extremely useful in constructing library interfaces, because
   certain conversions are often made, and the sheer repetitive
   boredom of installing correct argument count and type checking
   makes it tempting to forgo, leading to less stable code.

   define-safe-glue is usable at two levels:

     (1) "use" level
     (2) "extension" level

   the "extension" level is used to extend the functionality of the
   basic define-safe-glue, using a wrapper rewriter that passes
   hidden args to the underlying define-safe-glue.

   Using define-safe-glue
   ======================

   While the system is rife with examples, particularly in the
   corelib and unixm modules, perhaps a few more formal comments
   are appropriate.

   the usage template is:

   (define-safe-glue (function [arg-form ...] [#rest [rest-arg]])
      [:template]
      [literals: (literal-expr ...)]
      [envt: ((literal-expr ...) ...)]
   {
     C body
   })

   where:

      arg-form ::= name
                |  (name type)

      literal-expr ::=  (& tlvarname)
                    |   'scheme-datum
                    |   compile-time-const-expr

   In an arg-form, type may be either a regular class name 
   or a special type name recognized by the safe-glue (the list
   of recognized types can be extended -- see "Extending define-safe-glue",
   below).

   The first form of arg-form creates a macro that refers to the
   apropriate VM register, as usual for define-glue (this is important
   because you sometimes have to be careful to not stomp on your
   inputs when you're constructing your outputs).

   The second form, (foo bar) causes the macro name of the register
   to be "foo_raw", and creates
   one or more C local variables to hold the validated type.  Which
   variables are created depends on the type.  If the type is not
   specially recognized, then there is one variable of type `obj' called
   "foo".

   By convention, for specially recognized types, if there is a single
   transformation, it will be to an appropriately typed variable with
   the "foo" name.

   If the type specifies the construction of more than one local variable,
   the variables will be named "foo_blech" where blech describes
   the transformation applied to "raw_foo" to get the variable.

   (None of the basic recognized types expand to more than one
    variable, but the x11 package has many examples)

   The basic recognized types are:

      type name         safety predicate       type of local var
      <raw-int>            fixnum?                  int
      <raw-ascii-char>     ascii-char?              UINT_8
      <raw-string>         string?                  char *

   The basic implementation also emits optimized type checking
   for the types <fixnum> and <string>

   If :template is specified, then the value of the 'function'
   variable is a <template> instead of a closure, and 'envt:' cannot
   be specified

   [Note: 'envt:' is supported by define-glue, I'm not sure it
   gets passed through properly by define-safe-glue...]

   If #rest is specified, then the args are checked for a min
   value instead of an exact value.  If a name follows #rest, then
   the arguments are collected into a list in the usual fashion.
   Otherwise, the args are left alone (useful when it doesn't
   make sense to construct a list and then go cruising through
   it when the values can be accessed directly in the VM registers)

   Extending define-safe-glue
   ==========================

   In order to extend define-safe-glue, a rewriter is created
   which takes it's argument, adds a description of it's type
   extensions, and passes that to define-safe-glue.

   A template for doing this is:

	(define-rewriter (define-rstore-glue form)
	  (cons 'define-safe-glue
		(cons '(extend-known-types 
			    {type-extension ...}
			    {view-macro ...})
		      (cdr form))))

   define-safe-glue will recognize the extend-known-types argument
   and interpret it as an extention (note this makes it impossible
   to use define-safe-glue to create a function named extend-known-types)

   Type format for the type-extension is as follows:

	type-extension ::= (type-name recognition-style [view ...])
                        |  (view-macro-name var-decl-fmt
					    var-init-fmt
					    [local-name-fmt])

	recognition-style ::= (instance <foo-class>)
			   |  (class-eq? <foo-class>)
			   |  (primitive "OBJ_ISA_FOO_P" <foo-class>)

   the class-eq? recognition style requires an exact class match,
   and hence is faster but doesn't tolerate subclassing or non-PTR
   types.  the primitive recognition style can be used for 
   non-PTR types where subclasses are not permitted.

	view ::= (var-decl-fmt var-init-fmt [local-name-fmt])
              |  view-macro-name

   var-decl-fmt and var-init-fmt are format strings used to declare
   and initialize (respectively) the C variable that will hold the
   view of the argument's value.

   local-name-fmt is used to show how the base name is to be
   transformed into the view's destructured name.


   If there are no views given, then the variable name is a C macro
   for the input register.  Otherwise, as indicated above, the register
   is macro'd to "foo_raw", and each view creates and initializes
   a C local variable.

   Here are some complete examples of type-extensions from x11 package:

	an <X-font> is a gvec which contains in it's first slot
	a fixnum-looking pointer to the C XFontStruct.

	     (<X-font> 
	      (class-eq? <X-font>)
	      ("XFontStruct *~a"
	       "(XFontStruct *)OBJ_TO_RAW_PTR( gvec_read(~a,SLOT(0)) )"))

        An <X-window> (subclass of <X-drawable>) is a gvec which
        has a raw display pointer in slot 0 and an XID in slot 1.
        Because many objects have this basic structure, macros were
        created for these views.

             (<X-window> 
	      (class-eq? <X-window>) 
	      display/0 
	      xid/1)

        These are the actual macro definitions:

             (display/0
	      "Display *~a" 
	      "OBJ_TO_DISPLAY( gvec_read(~a,SLOT(0)) )"
	      "~a_dsp")

	     (xid/1 
	      "XID ~a"
	      "FX_TO_XID( gvec_read(~a,SLOT(1)) )"
	      "~a_xid")
|#


(define-rewriter (define-safe-glue form)
  (define known-types
    ;;
    ;; [0] recognized type name OR view name
    ;; [1] type-checking method ((primitive predicate required-class), 
    ;;                           (class-eq? class), or 
    ;;                           (instance? class))
    ;; [2...] "views"
    ;;		a view is a way of extracting C-style data from
    ;;		the raw (scheme) argument.  This includes the
    ;;		variable declaration, the extractor function, 
    ;;		and an option name rewriting.  All of these are
    ;;		'format' argument strings.
    ;;
    '((<raw-int> (primitive "OBJ_ISA_FIXNUM" <fixnum>)
		 ("int ~a" "fx2int(~a)"))
      (<fixnum> (primitive "OBJ_ISA_FIXNUM" <fixnum>))
      (<string> (primitive "STRING_P" <string>))
      (<ascii-char> (primitive "OBJ_ISA_ASCII_CHAR" <ascii-char>))
      (<<class>> (primitive "CLASS_P" <<class>>))
      (<allocation-area> (class-eq? <allocation-area>)
			 ("AllocArea *~a"
			  "(AllocArea *)PTR_TO_DATAPTR(~a)"))
      (<raw-ascii-char> (primitive "OBJ_ISA_ASCII_CHAR" <ascii-char>)
			("UINT_8 ~a" "ASCII_CHAR_VALUE(~a)"))
      (<raw-string> (primitive "STRING_P" <string>)
		    ("char *~a" "(char *)string_text(~a)"))))
  ;;
  (if (and (pair? (cadr form))
	   (eq? (caadr form) 'extend-known-types))
      (begin
	;(format #t "EXTEND-KNOWN-TYPES: ~s\n" (cadr form))
	;(for-each (lambda (t) (format #t "\tIE: ~s\n" t)) (cdadr form))
	(set! known-types (append (cdadr form) known-types))
	;(format #t "introducing extended types: (NEW LIST)\n")
	;(for-each (lambda (t) (format #t "\t~s\n" t)) known-types)
	(set! form (cdr form))))
  ;;
  ;;
  (let ((name (caadr form))
	(args (cdadr form))
	(body (cddr form))
	(literals '())
	(template #f)
	(subsequents '())
	(rest-arg #f))
    ;;
    ;; check for a #rest arg
    ;;
    (if (memq '#rest args)
	(begin
	  (if (null? (cdr (memq '#rest args)))
	      ;;
	      ;; (foo bar baz #rest) syntax
	      ;;
	      (begin
		(set! args (reverse (cdr (reverse args))))
		(set! rest-arg #t))
	      ;;
	      ;; (foo bar baz #rest somelst) syntax
	      ;;
	      (begin
		(set! rest-arg (last args))
		(set! args (reverse (cddr (reverse args))))))))
    ;;
    ;;
    (define (alloc-literal lit)
      (let ((i (length literals)))
	;(format #t "alloc-literal(~d) = ~s\n" i lit)
	(set! literals
	      (append literals (list lit)))
	i))
    ;;
    (define (expand-view view)
      ;(format #t "expanding view: ~s\n" view)
      (if (symbol? view)
	  (let ((a (assq view known-types)))
	    (if a
		(cdr a)
		(syntax-error "view macro `~a' undefined" view)))
	  view))
    ;;
    (define (render-isa-check raw-arg arg check-expr expect)
      (format #t "if (!~a)\n" check-expr)
      (format #t "   scheme_error( string_text(LITERAL(~d)), 1, ~a );\n"
		  (alloc-literal 
		   (format #f "~a: bad arg ~s == ~~s (expected ~a)"
			   name
			   arg
			   expect))
		  raw-arg))
    ;;
    (define (get-type-methods type-name)
      (let ((m (assq type-name known-types)))
	(or m (list type-name (list 'instance? type-name)))))
    ;;
    (define (view-name base-name view)
      (if (> (length view) 2)
	  (format #f (caddr view) base-name)
	  base-name))
    ;;
    (define (render-var-decl name views)
      (if (null? views)
	  (format #t "obj ~a;\n" name)
	  (for-each (lambda (v)
		      (format #t (car v) (view-name name v))
		      (format #t ";\n"))
		    views)))
    ;;
    (define (render-type-check raw-name new-name type-methods)
      (let ((guard (cadr type-methods))
	    (views (map expand-view (cddr type-methods))))
	;;
	;(format #t "views: ~s\n" views)
	;;
	(case (car guard)
	  ((primitive)
	   (render-isa-check
	    raw-name 
	    new-name
	    (format #f "~a(~a)" (cadr guard) raw-name)
	    (caddr guard)))
	  ((instance?)
	   (render-isa-check
	    raw-name
	    new-name
	    (format #f "instance_p(~a,TLREF(~d))"
		    raw-name
		    (alloc-literal (list '& (cadr guard))))
	    (cadr guard)))
	  ((class-eq?)
	   (render-isa-check 
	    raw-name
	    new-name
	    (format #f "OBJ_ISA_PTR_OF_CLASS(~a,TLREF(~d))"
		    raw-name
		    (alloc-literal (list '& (cadr guard))))
	    (cadr guard)))
	  (else
	   (syntax-error "invalid type method (guard): ~s" guard)))
	;;
	(if (null? views)
	    (format #t "~a = ~a;\n" new-name raw-name)
	    (for-each (lambda (v)
			(format #t "~a = " (view-name new-name v))
			(format #t (cadr v) raw-name)
			(format #t ";\n"))
		      views))))
    ;;
    ;; parse a `:template' flag
    ;;
    (if (eq? (car body) ':template)
	(begin
	  (set! body (cdr body))
	  (set! template #t)))
    ;;
    ;; strip off any literals used
    ;;
    (if (eq? (car body) 'literals:)
	(begin
	  (set! literals (cadr body))
	  (set! subsequents (cdddr body))
	  (set! body (caddr body)))
	(begin
	  (set! subsequents (cdr body))
	  (set! body (car body))))
    ;;
    ;(format #t "base literals: ~s\n" literals)
    (let ((raw-args (map (lambda (a)
			   (if (pair? a)
			       (string->symbol
				(string-append "raw_"
					       (symbol->string (car a))))
			       a))
			 args)))
      ;(format #t "raw args: ~s\n" raw-args)
      
      (let ((p (open-output-string)))
	;(format #t "extended literals: ~s\n" literals)
	(with-output-to-port p
	  (lambda ()
	    ;;
	    ;; render declarations
	    ;;
	    (for-each (lambda (a ra)
			(if (pair? a)
			    (render-var-decl 
			     (car a) 
			     (map expand-view
				  (cddr (get-type-methods (cadr a)))))))
		      args
		      raw-args)
	    (if (symbol? rest-arg)
		(format #t "obj ~a;\n" rest-arg))
	    ;;
	    ;; emit basic preamble
	    ;;
	    (if rest-arg
		(begin
		  (format #t "COUNT_ARGS_AT_LEAST(~d);\n" (length args))
		  (if (symbol? rest-arg)
		      ;;
		      ;; collect 'em and put em in a var
		      ;; (only works for <10 fixed args)
		      ;;
		      (begin
			(format #t "COLLECT~d();\n" (length args))
			(format #t "~a = REG~d;\n" rest-arg (length args)))))
		(format #t "COUNT_ARGS(~d);\n" (length args)))
	    ;;
	    ;; emit extended (type checking & conversion) preamble
	    ;;
	    (for-each (lambda (a ra)
			(if (pair? a)
			    (render-type-check ra
					       (car a)
					       (get-type-methods (cadr a)))))
		      args
		      raw-args)
	    ;;
	    ;; emit the body
	    ;;
	    (write-char #\{)
	    (display body)
	    (write-char #\})))
	(append '(define-glue)
		(list (cons name raw-args))
		(if template
		    '(:template)
		    '())
		(list 'literals: literals)
		(list (string->c-text (close-output-port p)))
		subsequents)))))
