/*
 * dmnt.c - DEC OSF/1 mount support functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dmnt.c,v 1.11 95/12/07 15:44:57 abe Exp $";
#endif


#include "lsof.h"


/*
 * Local static information
 */

#undef	KERNEL
#include <sys/fs_types.h>		/* this defines char *mnt_names[] */


/*
 * readmnt() - read mount table
 */

int
readmnt()
{
	char *dn = NULL;
	int err = 0;
	char *ln;
	struct statfs *mb;
	struct mounts *mtp;
	int n;
	int procfs = 0;
	struct stat sb;
/*
 * Access mount information.
 */
	if ((n = getmntinfo(&mb, MNT_NOWAIT)) <= 0) {
		(void) fprintf(stderr, "%s: no mount information\n", Pn);
		return(0);
	}
/*
 * Read mount information.
 */
	for (; n; n--, mb++) {
		if (mb->f_type == MOUNT_NONE || mb->f_type >= MOUNT_MAXTYPE)
			continue;
	/*
	 * Avoid file systems under automounter control if they're not
	 * currently mounted.
	 */
		if (mb->f_type == MOUNT_NFS) {

		/*
		 * The mount-from name of some unmounted file systems under
		 * automounter control end with ``:(pid<n>):'' -- where <n>
		 * is the PID of the automounter process.
		 */
		    if ((ln = strchr(mb->f_mntfromname, ':')) != NULL) {
			if (strncmp(ln+1, "(pid", 4) == 0 && isdigit(*(ln+5)))
			{
			    for (ln += 6; *ln && isdigit(*ln); ln++)
				;
			    if (*ln == ')' && *(ln+1) == '\0')
				continue;
			}
		    }
		/*
		 * Another automounter mount-from name form is "amd:<n>" --
		 * where <n> is the PID of the automounter process.
		 */
		    if (strncmp(mb->f_mntfromname, "amd:", 4) == 0
		    &&  isdigit(mb->f_mntfromname[4])) {
			ln = &mb->f_mntfromname[5];
			while (*ln && isdigit(*ln))
			    ln++;
			if (*ln == ':' && *(ln+1) == '\0')
			    continue;
		    }
		}
	/*
	 * Interpolate a possible symbolic directory link.
	 */
		if (dn)
			(void) free((FREE_P *)dn);
		if ((dn = (char *)malloc((MALLOC_S)strlen(mb->f_mntonname)+1))
		== NULL) {
			err = 1;
			break;
		}
		(void) strcpy(dn, mb->f_mntonname);
		if ((ln = Readlink(dn)) == NULL) {
		    if (!Fwarn) {
			(void) fprintf(stderr,
			    "      Output information may be incomplete.\n");
			}
			err = 2;
			continue;
		}
		if (ln != dn) {
			(void) free((FREE_P *)dn);
			dn = ln;
		}
	/*
	 * Stat() the directory.
	 */
		if (statsafely(dn, &sb)) {
		    if (!Fwarn) {
			(void) fprintf(stderr,
			    "%s: WARNING: can't stat() %s file system %s\n",
			    Pn, mnt_names[mb->f_type],
			    mb->f_mntonname);
			(void) fprintf(stderr,
			    "      Output information may be incomplete.\n");
		    }
		    err = 2;
		    if (mb->f_type != MOUNT_PROCFS && mb->f_type != MOUNT_MSFS)
		    {
			memset((char *)&sb, 0, sizeof(sb));
			sb.st_dev = (dev_t)mb->f_fsid.val[0];
			sb.st_mode = S_IFDIR | 0777;
			if (!Fwarn)
			    (void) fprintf(stderr,
				"      assuming dev=%x from mount table\n",
				sb.st_dev);
		    }
		}
	/*
	 * Allocate and fill a local mount structure.
	 */
		if ((mtp=(struct mounts *)malloc(sizeof(struct mounts))) == NULL
		||  (mtp->fsname = (char *)malloc((MALLOC_S)
			strlen(mb->f_mntfromname)+1)) == NULL)
		{
			err = 1;
			break;
		}
		(void) strcpy(mtp->fsname, mb->f_mntfromname);
		mtp->dir = dn;
		dn = NULL;
		mtp->dev = sb.st_dev;
		mtp->fsid = mb->f_fsid;
		mtp->inode = sb.st_ino;
		mtp->mode = sb.st_mode;
		mtp->next = Mtab;
		mtp->rdev = sb.st_rdev;
		if (mb->f_type == MOUNT_PROCFS) {

		/*
		 * Save information on exactly one procfs file system.
		 */
			if (procfs)
					Mtprocfs = NULL;
				else {
					procfs = 1;
					Mtprocfs = mtp;
				}
		}
		Mtab = mtp;
	}
/*
 * Handle errors.
 */
	switch(err) {
	case 1:
		(void) fprintf(stderr, "%s: no space for mount at %s (%s)\n",
			Pn, mb->f_mntonname, mb->f_mntfromname);
		return(0);
	case 2:
		return(1);
	}

	return(1);
}


/*
 * readvfs() - read vfs structure
 */

struct l_vfs *
readvfs(vm)
	struct mount *vm;		/* mount address in vnode */
{
	struct mount m;
	struct l_vfs *vp;
	fsid_t f;
	struct mounts *mp;

#if	_OSF1V>=40000
	int bl;
	char fb[MAX_MNT_PATHLEN+1];
	char ob[MAX_MNT_PATHLEN+1];
#endif	/* _OSF1V>=40000 */

/*
 * Search for match on existing entry.
 */
	for (vp = Lvfs; vp; vp = vp->next) {
		if (vm == vp->addr)
			return(vp);
	}
/*
 * Read the (new) mount structure, allocate a local entry, and fill it.
 */
	if (kread((KA_T)vm, (char *)&m, sizeof(m)) != 0)
		return(NULL);
	if ((vp = (struct l_vfs *)malloc(sizeof(struct l_vfs))) == NULL) {
		(void) fprintf(stderr, "%s: PID %d, no space for vfs\n",
			Pn, Lp->pid);
		exit(1);
	}

#if	_OSF1V<40000
	if ((vp->dir = (char *)malloc(strlen(m.m_stat.f_mntonname)+1))
	== NULL
	||  (vp->fsname = (char *)malloc(strlen(m.m_stat.f_mntfromname)+1))
	== NULL)
#else	/* _OSF1V>=40000 */
	bl = sizeof(ob) - 1;
	if (!m.m_stat.f_mntonname
	||  kread((KA_T)m.m_stat.f_mntonname, ob, bl))
		bl = 0;
	ob[bl] = '\0';
	bl = sizeof(fb) - 1;
	if (!m.m_stat.f_mntfromname
	||  kread((KA_T)m.m_stat.f_mntfromname, fb, bl))
		bl = 0;
	fb[bl] = '\0';
	if ((vp->dir = (char *)malloc(strlen(ob)+1))
	== NULL
	||  (vp->fsname = (char *)malloc(strlen(fb)+1))
	== NULL)
#endif	/* _OSF1V<40000 */

	{
		(void) fprintf(stderr, "%s: PID %d, no space for mount names\n",
			Pn, Lp->pid);
		exit(1);
	}

#if	_OSF1V<40000
	(void) strcpy(vp->dir, m.m_stat.f_mntonname);
	(void) strcpy(vp->fsname, m.m_stat.f_mntfromname);
#else	/* _OSF1V>=40000 */
	(void) strcpy(vp->dir, ob);
	(void) strcpy(vp->fsname, fb);
#endif	/* _OSF1V<40000 */

	vp->addr = vm;
	vp->fsid = m.m_stat.f_fsid;
	vp->type = m.m_stat.f_type;

#if	defined(HASFSINO)
	vp->fs_ino = 0;
#endif	/* defined(HASFSINO) */

	vp->next = Lvfs;
/*
 * Derive the device and raw device numbers from a search for the
 * file system ID in the local mount table.
 */
	vp->dev = vp->rdev = 0;
	for (f = vp->fsid, mp = Mtab; mp; mp = mp->next) {
		if (f.val[0] == mp->fsid.val[0]
		&&  f.val[1] == mp->fsid.val[1])
		{
			vp->dev = mp->dev;
			vp->rdev = mp->rdev;

#if	defined(HASFSINO)
			vp->fs_ino = mp->inode;
#endif	/* defined(HASFSINO) */

			break;
		}
	}
	return(vp);
}
