;**************************************************************************
;*
;* Boot-ROM-Code to load an operating system across a TCP/IP network.
;*
;* Module:  dosinit.asm
;* Purpose: Initialize the DOS simulator, setup all interupt vectors etc.
;* Entries: _init_minidos, dosfatal, call21
;*
;**************************************************************************
;*
;* Copyright (C) 1995,1996 Gero Kuhlmann <gero@gkminix.han.de>
;*
;*  This program is free software; you can redistribute it and/or modify
;*  it under the terms of the GNU General Public License as published by
;*  the Free Software Foundation; either version 2 of the License, or
;*  any later version.
;*
;*  This program is distributed in the hope that it will be useful,
;*  but WITHOUT ANY WARRANTY; without even the implied warranty of
;*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;*  GNU General Public License for more details.
;*
;*  You should have received a copy of the GNU General Public License
;*  along with this program; if not, write to the Free Software
;*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;*


;
;**************************************************************************
;
; Include assembler macros:
;
include ..\..\headers\asm\macros.inc
include ..\..\headers\asm\layout.inc
include ..\..\headers\asm\memory.inc
include .\dospriv.inc


;
;**************************************************************************
;
; The DOS simulator needs it's own stack when int21h is called. This stack
; is placed in the BSS segment. Note that the real DOS uses three different
; stacks. This will not be completely simulated. The stack for function
; 59h is not necessary, so we only simulate two different stacks. The
; stack signature is contained in the registers structure, so this struc-
; ture has to come before the stack itself.
;
bss_start				; start BSS segment

		public	dos1_regs
		public	dos2_regs

STACKSIZE	equ	512		; pretty small, so be careful.
STACKSIG	equ	0AA55h		; signature for checking stack overflow

dos1_regs	old_reg	<>
dos1_stack	db	STACKSIZE dup (?)

dos2_regs	old_reg	<>
dos2_stack	db	STACKSIZE dup (?)

temp_bp		dw	?		; temporary storage for BP
temp_ax		dw	?		; temporary storage for AX
temp_bx		dw	?		; temporary storage for BX

bss_end


;
;**************************************************************************
;
; And now some variables for our mini-DOS simulator:
;
data_start				; start data segment

		extrn	curr_psp:word	; segment of current psp
		extrn	first_mcb:word	; segment of first MCB in list

		public	dos_active

dos_active	db	0		; DOS active counter
prognum		db	0		; current program number

data_end


;
;**************************************************************************
;
; Define macros to produce jump table entries.
;
tab	macro	funcno

	db	0&funcno&h			;; save function number
if 0&funcno&h ne 0FFh				;; 0FFh means end of table
	dw	offset cgroup:dos&funcno	;; save pointer to label

	extrn	dos&funcno:near			;; define label as near external
endif

	endm

TABSIZE	equ	3			; number of bytes per table entry


;
;**************************************************************************
;
; Start code segment.
;
text_start

	public	_init_minidos		; define entry points
	public	call21
	public	dosfatal

; External routines:

	extrn	loadprog:near

; General library functions:

	extrn	_fatal:near
	extrn	prnstr:near
	extrn	prnbyte:near


;
;**************************************************************************
;
; Interrupt handler table.
;
inttab	dw	20h * 4, offset cgroup:int20	; terminate program
	dw	21h * 4, offset cgroup:int21	; dos function scheduler
	dw	27h * 4, offset cgroup:int27	; terminate and stay resident
	dw	29h * 4, offset cgroup:int29	; print character

FRSTINT	equ	20h				; first interrupt used by DOS
LSTINT	equ	2Fh				; last interrupt used by DOS
INTNUM	equ	($ - inttab) / 4		; number of interrupts in table


;
;**************************************************************************
;
; Initialize the mini-DOS simulator module.
; Input:  1. Arg on stack  -  far pointer to program pointer table
;         2. Arg on stack  -  far pointer to first usable memory address
; Output: None
;
_init_minidos	proc	near

	penter					; setup standard stack frame
	push	es
	push	di
	mov	dos1_regs.stack_sig,STACKSIG	; sig for stack overflow check
	mov	dos2_regs.stack_sig,STACKSIG	; sig for stack overflow check

; Initialize DOS memory by setting up the first memory control block.

	getarg	ax,2			; get second argument: offset
	getarg	bx,3			; get second argument: segment
	shift	shr,ax,4		; adjust pointer to nearest paragraph
	add	ax,bx			; boundary
	add	ax,2			; one additional paragraph for safety
	mov	bx,LOWMEM
	sub	bx,ax			; check that we have enough memory
	jc	short init3
	cmp	bx,RAMSIZE / 16
	ja	short init4
init3:	mov	ax,FATAL_NOMEM		; fatal error -> not enough memory
	jmp	dosfatal

init4:	dec	bx			; MCB requires one paragraph of free
	push	ds			; memory
	mov	first_mcb,ax		; save pointer to first MCB
	mov	ds,ax
	mov	ds:[mcb_id],MCB_END_ID
	mov	ds:[mcb_owner],0	; set MCB of first free memory block
	mov	ds:[mcb_size],bx
	pop	ds

; Install interrupt handlers.

	cli
	xor	ax,ax
	mov	es,ax
	mov	di,FRSTINT * 4		; first interrupt is 20h
	mov	ax,offset cgroup:unused
	mov	cx,LSTINT - FRSTINT + 1	; determine number of interrupt vectors
init1:	mov	es:[di + 0],ax		; save pointer to 'unused' routine
	mov	es:[di + 2],cs
	add	di,4			; go to next interrupt vector
	loop	short init1

	mov	bx,offset cgroup:inttab	; pointer to interrupt table
	mov	cx,INTNUM
init5:	mov	di,cs:[bx + 0]		; get pointer to interrupt vector
	mov	ax,cs:[bx + 2]
	mov	es:[di + 0],ax		; save offset
	mov	es:[di + 2],cs		; then segment
	add	bx,4			; continue with next interrupt
	loop	short init5
	sti

; Initialize the video screen. This ensures that the BIOS didn't leave
; us with some unusual video modes.

	mov	ah,0Fh
	int	10h
	cmp	al,2			; check for modes 2, 3 or 7, e.g.
	je	short init6		; 80x25 text mode
	cmp	al,3
	je	short init6
	cmp	al,7
	je	short init6
	mov	ax,0003h		; set screen to mode 3
	int	10h

; Call all DOS programs sucessively

init6:	getarg	di,0			; get address to program list from
	getarg	es,1			; stack
init7:	mov	ax,es:[di+0]		; get far pointer to binary image
	or	ax,ax
	jz	short init8		; check for end of list
	mov	si,es:[di+2]		; get far pointer to command line
	add	di,4
	inc	prognum
	push	di
	mov	di,ax
	call	loadprog		; load and run the program
	pop	di
	jmp	short init7

init8:	pop	di
	pop	es			; that's it
	pleave
	ret

_init_minidos	endp


;
;**************************************************************************
;
; Handle fatal errors by printing a message and then rebooting.
; Input:  AX  -  error number
; Output: routine does not return
;
dosfatal	proc	near

	assume	ds:dgroup
	assume	ss:nothing

	sti
	push	ax
	mov	ax,NEWDATA
	mov	ds,ax
	mov	bx,offset cgroup:errmsg
	call	prnstr			; print starting message
	pop	si
	push	si
	shl	si,1
	mov	bx,cs:errtab[si]	; print error message
	call	prnstr
	pop	ax
	cmp	ax,FATAL_PROG
	jne	short fatal9
	mov	al,prognum		; print program number
	call	prnbyte
fatal9:	jmp	_fatal

; Error messages

errmsg	db	0Dh,0Ah
	db	'DOS ERROR: '
	db	0

; The following table has to be identical with the error numbers in
; the dospriv include file:

errtab	dw	offset cgroup:errnom	; not enough memory
	dw	offset cgroup:errstk	; stack overflow
	dw	offset cgroup:errnop	; no active process
	dw	offset cgroup:errmem	; memory corrupt
	dw	offset cgroup:errprg	; program error

errnom	db	'no free mem',0
errstk	db	'stack overflow',0
errnop	db	'inv term',0
errmem	db	'mem corrupt',0
errprg	db	'corrupt prog ',0

dosfatal	endp


;
;**************************************************************************
;
; Jump table for main dispatcher interrupt. Put it into code segment
; (instead const segment) to save space in lower ram area.
;
jmptab	label	byte

; functions for basic I/O:
	irp	x,<01,02,03,04,05,06,07,08,09,0A,0B,0C>
	tab	<&x>
	endm

; functions for system management:
	irp	x,<0D,0E,19,1A,25,2E,2F,30,34,35,37,54>
	tab	<&x>
	endm

; functions for time operations:
ifdef NEEDTIME
	irp	x,<2A,2B,2C,2D>
	tab	<&x>
	endm
endif

; functions for memory and process management:
	irp	x,<00,31,48,49,4A,4C,4D>
	tab	<&x>
	endm

; functions for file handling:
	irp	x,<3C,3D,3E,3F,40>
	tab	<&x>
	endm

	tab	<FF>			; end of table

; Those functions which were only implemented before DOS2.0 return
; a different error code than those which were introduced with
; DOS2.0 and later. The first "later" function is 30h:

DOS20FN	equ	30h


;
;**************************************************************************
;
; Routine for main dispatcher interrupt 21h. This involves changing to a
; new stack, and then searching for the required function in the function
; table. The function handler has to get the caller's data segment from
; old_ds if necessary.
; Input:  AH  -  DOS function number
;         others depending on function number
; Output: depending on function number
;
int21		proc	far

	assume	ds:dgroup
	assume	ss:nothing

; First check for functions which don't use an internal stack.

int21s:	push	ds
	push	ax
	mov	ax,NEWDATA		; switch to new data segment
	mov	ds,ax
	pop	ax
	cmp	ah,33h			; function 33h is not implemented
	je	short nostkr
	cmp	ah,64h			; function 64h is not implemented
	je	short nostkr
	cmp	ah,59h
	jne	short isno59
	mov	ax,00FFh		; function 59h is not implemented,
	mov	bx,0D05h		; so just return some dummy values
	mov	ch,01h
	jmp	short nostkr
isno59:	cmp	ah,50h
	jne	short isno50
	mov	curr_psp,bx		; implement function 50h
	jmp	short nostkr
isno50:	cmp	ah,51h
	je	short isf51		; function 51h is the same as 62h
	cmp	ah,62h
	jne	short int21r
isf51:	mov	bx,curr_psp		; implement function 51h and 62h
nostkr:	pop	ds
	iret

; For all other functions set new stack and then check in function table
; for the function handler. Select the correct stack for the function
; used. If changing anything in this code remember to also change the
; basicio functions!

int21r:	cli
	mov	temp_bp,bp
	mov	bp,offset dgroup:dos1_regs
	or	ah,ah
	jz	short int21v
	cmp	ah,0Ch				; select correct stack
	jbe	short int21u
int21v:	mov	bp,offset dgroup:dos2_regs
int21u:	pop	ds:old_ds[bp]
	push	ax
	mov	ax,sp
	add	ax,2
	push	ax
	pop	word ptr ds:[old_stack + 0][bp]
	mov	word ptr ds:[old_stack + 2][bp],ss
	pop	temp_ax
	mov	temp_bx,bx
	mov	ax,temp_bp
	mov	ds:old_bp[bp],ax
	mov	ax,ds
	mov	bx,bp
	add	bx,size old_reg + STACKSIZE - 2
	mov	ss,ax				; set new stack
	mov	sp,bx
	assume	ss:dgroup

; Search for the function in the function table.

	mov	bx,offset cgroup:jmptab
	mov	ax,temp_ax
int21l:	mov	al,cs:[bx]
	cmp	al,0FFh			; check for end of table
	je	short notfnd
	cmp	al,ah			; found function value?
	je	short int21f
	add	bx,TABSIZE		; not found, loop to next
	jmp	short int21l		; table entry

; Found table entry, so prepare registers and jump to the routine. This
; is done by putting it's address onto the stack, and then executing a
; near return instruction.

int21f:	inc	dos_active		; increment active counter
ifdef IS186
	push	offset cgroup:int21t	; push return address onto stack
else
	mov	ax,offset cgroup:int21t	; push return address onto stack
	push	ax
endif
	push	cs:[bx+1]		; push jump address onto stack
	mov	bx,temp_bx
	mov	ax,temp_ax		; restore all registers except DS
	sti
	retn				; jump to handler routine

; Return to the caller by restoring the stack without changing any
; registers. Also check for stack overflow. Take care so that the
; flags don't get changed.

int21t:	cli
	pushf
	dec	dos_active		; decrement active counter
	cmp	stack_sig[bp],STACKSIG	; check for stack overflow
	je	short noerr
	mov	ax,FATAL_STACK		; call fatal error handler
	jmp	dosfatal

noerr:	mov	temp_bx,bx
int21e:	mov	temp_ax,ax
	pop	ax			; get flags
	assume	ss:nothing
	mov	ss,word ptr ds:[old_stack + 2][bp]
	mov	sp,word ptr ds:[old_stack + 0][bp]
	mov	bx,sp
	mov	byte ptr ss:[bx+4],al	; this is the place where the INT
	mov	bx,temp_bx		; operation did put the flags
	mov	ax,temp_ax
	push	ds:old_bp[bp]
	mov	ds,ds:old_ds[bp]	; restore caller's registers
	pop	bp
	iret

; We couldn't find the requested function in the function table, so return
; with an error code in AX and the carry flag set. However, if it is a
; FCB related function, return with the proper error code for that.

	assume	ss:dgroup

notfnd:	cmp	ah,DOS20FN		; check if it's an old function
	jae	short notfn1
	mov	al,0FFh			; return with proper error code
	jmp	short notfn2

notfn1:	mov	ax,ERR_NOFN		; return error code
notfn2:	stc				; set carry flag to indicate error
	pushf
	jmp	short int21e

int21		endp


;
;**************************************************************************
;
; INT20h handler: terminate current program.
; Input:  none
; Output: routine does not return
;
int20		proc	far

	assume	ds:nothing
	assume	es:nothing
	assume	ss:nothing

	xor	ah,ah
	jmp	int21s			; just call DOS function 00h

int20		endp


;
;**************************************************************************
;
; INT27h handler: terminate current program and stay resident.
; Input:  DX  -  number of bytes in program memory block
; Output: routine does not return
;
int27		proc	far

	assume	ds:nothing
	assume	es:nothing
	assume	ss:nothing

	add	dx,000Fh
	rcr	dx,1			; round byte count to nearest
	shift	shr,dx,3		; paragraph
	mov	ah,31h			; just call DOS function 31h
	jmp	int21s

int27		endp


;
;**************************************************************************
;
; INT29h handler: print character onto screen
; Input:  AL  -  character to print
; Output: none
;
int29		proc	far

	assume	ds:nothing
	assume	es:nothing
	assume	ss:nothing

	push	ax
	push	bx
	mov	ah,0Eh
	mov	bx,0007h		; just print using the BIOS
	int	10h
	pop	bx
	pop	ax
	iret

int29		endp


;
;**************************************************************************
;
; Handler for unused interrupts
; Input:  none
; Output: none
; Registers changed: none
;
unused		proc	far

	assume	ds:nothing
	assume	es:nothing
	assume	ss:nothing

	iret

unused		endp


;
;**************************************************************************
;
; Another way for a program to call the function dispatcher is to call
; address psp:0005h. This will jump to the following address.
;
call21		proc	far

	assume	ds:nothing
	assume	es:nothing
	assume	ss:nothing

	cli
	push	bp
	mov	bp,sp
	push	ax
	mov	ax,[bp + 6]		; get return address to caller
	mov	[bp + 2],ax		; and put it into place
	pushf
	pop	[bp + 6]		; replace the old return address
	pop	ax			; with the flags. this makes the
	pop	bp			; call look like an INT.
	jmp	int21s

call21		endp

;
;**************************************************************************
;
text_end

	end

