;**************************************************************************
;*
;* Boot-ROM-Code to load an operating system across a TCP/IP network.
;*
;* Module:  utility.asm
;* Purpose: Various functions for the image loader routines, which are
;*          better written in assembler
;* Entries: _convmem, _extmem, _lmove, _exec_image
;*
;**************************************************************************
;*
;* Copyright (C) 1995,1996 Gero Kuhlmann <gero@gkminix.han.de>
;*
;*  This program is free software; you can redistribute it and/or modify
;*  it under the terms of the GNU General Public License as published by
;*  the Free Software Foundation; either version 2 of the License, or
;*  any later version.
;*
;*  This program is distributed in the hope that it will be useful,
;*  but WITHOUT ANY WARRANTY; without even the implied warranty of
;*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;*  GNU General Public License for more details.
;*
;*  You should have received a copy of the GNU General Public License
;*  along with this program; if not, write to the Free Software
;*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;*


;
;**************************************************************************
;
; Include assembler macros:
;
include ..\..\headers\asm\macros.inc
include ..\..\headers\asm\layout.inc
include ..\..\headers\asm\memory.inc


;
;**************************************************************************
;
; Layout of descriptors in GDT:
;
gdt		struc

gdt_length	dw	?		; length of segment
gdt_base	db	3 dup (?)	; base of segment
gdt_flags	db	?		; descriptor flags
gdt_res		dw	?		; reserved

gdt		ends

gdt_stdflag	equ	93h		; standard flags for GDT entries


;
;**************************************************************************
;
; BSS segment
;
bss_start

gdt_table	gdt	6 dup (<>)	; descriptor table for move routine

bss_end


;
;**************************************************************************
;
; Start code segment.
;
text_start

	public	_convmem		; define entry points
	public	_extmem
	public	_lmove
	public	_exec_image

	extrn	_fatal:near


;
;**************************************************************************
;
; Get amount of conventional memory in bytes.
; Input:  none
; Output: memory amount
;
_convmem	proc	near

	int	12h			; just call the BIOS
	jmp	short extm1

_convmem	endp


;
;**************************************************************************
;
; Get amount of extended memory in bytes.
; Input:  none
; Output: memory amount
;
_extmem		proc	near

	mov	ah,88h
	int	15h			; just call the BIOS
extm1:	mov	dx,ax
	shift	shr,dx,6		; multiply value with 1024 to get
	shift	shl,ax,10		; number of bytes
	ret

_extmem		endp


;
;**************************************************************************
;
; Move a memory block into extended memory.
; Input:  1. arg  -  linear address to destination (long word)
;         2. arg  -  near pointer to source memory block
;         3. arg  -  length of memory block
; Output: none
;
_lmove		proc	near

	penter
	getarg	bx,3
ifdef IS386
	getarg	edx,0			; check for copy into lower memory
	cmp	edx,00100000h
else
	getarg	ax,0
	getarg	dx,1			; check for copy into lower memory
	cmp	dx,0010h
endif
	jae	short lmove1

; Handle moves into lower memory

	cld
	push	es
	push	si
	push	di
ifdef IS386
	mov	di,dx
	shr	edx,4			; convert linear address into seg:ofs
else
	mov	di,ax
	shift	shr,ax,4		; convert linear address into seg:ofs
	shift	shl,dx,12
	or	dx,ax
endif
	and	di,000Fh		; load destination pointer
	mov	es,dx
	getarg	si,2			; load source pointer
	mov	cx,bx
	inc	cx			; convert byte count into word count
	shr	cx,1
	rep	movsw			; do the copy
	pop	di
	pop	si
	pop	es
	jmp	short lmove9

; Handle moves into high memory

lmove1:
ifdef IS386
	mov	dword ptr gdt_table[3 * type gdt].gdt_base[0],edx
else
	mov	word ptr gdt_table[3 * type gdt].gdt_base[0],ax
	mov	gdt_table[3 * type gdt].gdt_base[2],dl
endif
	mov	gdt_table[3 * type gdt].gdt_flags,gdt_stdflag
	mov	gdt_table[3 * type gdt].gdt_length,bx
	mov	gdt_table[3 * type gdt].gdt_res,0
ifdef IS386
	mov	eax,ds
	shl	eax,4
	getarg	edx,2,word
	add	eax,edx
	mov	dword ptr gdt_table[2 * type gdt].gdt_base[0],eax
else
	mov	ax,ds
	shift	rol,ax,4
	mov	cx,ax
	and	ax,0FFF0h		; convert segment and offset into
	and	cx,0000Fh		; a linear address
	getarg	dx,2
	add	ax,dx
	adc	cx,0
	mov	word ptr gdt_table[2 * type gdt].gdt_base[0],ax
	mov	gdt_table[2 * type gdt].gdt_base[2],cl
endif
	mov	gdt_table[2 * type gdt].gdt_flags,gdt_stdflag
	mov	gdt_table[2 * type gdt].gdt_length,bx
	mov	gdt_table[2 * type gdt].gdt_res,0

	push	es
	push	si
	mov	cx,bx
	inc	cx			; convert byte count into word count
	shr	cx,1
	mov	ax,ds
	mov	es,ax
	mov	si,offset dgroup:gdt_table
	mov	ah,87h
	int	15h			; call BIOS to do the copy
	pop	si
	pop	es
lmove9:	pleave
	ret

_lmove		endp


;
;**************************************************************************
;
; Execute the boot image
; Input: 1. arg  -  header mode
;        2. arg  -  far pointer to execution point
;        3. arg  -  far pointer to header
;        4. arg  -  far pointer to parameter block
; Output: Routine does not return
;
_exec_image	proc	near

	penter

; First call the interrupt 2Fh to deinitialize all modules which might
; be loaded and which redirected any interrupts. The magic number is
; indeed what the name means: magic - you can find it in programs like
; pktwatch.

	mov	ax,0CF45h
	int	2Fh

; Reset the interrupt vector table. We don't need to save
; all segment and index registers since this routine will
; never return to the C code.

	cli				; make this copying atomic
	cld
	xor	ax,ax
	mov	es,ax
	mov	ax,OLDINTS
	mov	ds,ax
	xor	si,si
	xor	di,di
	mov	cx,VECTSIZE / 2
	rep	movsw
	sti

	getarg	ax,0
	cmp	ax,1			; check for mode 1
	jne	short exec1

; Handle mode number 1. This means copying the header block to
; the execution point, and setting the stack just below there.
; Note that the image doesn't get anything on it's stack. It
; would be too complicated to check for overwriting the bootp
; block and/or stack with the header block.

	getarg	es,2			; get the pointer to the boot
	getarg	di,1			; area into ES:DI
	mov	bx,di
	getarg	ds,4			; put the pointer to the header
	getarg	si,3			; block into DS:SI
	mov	cx,512 / 2
	rep	movsw			; now place the boot block into
	cli				; it's final place
	mov	ax,es
	mov	sp,bx
	mov	ss,ax			; set new stack
	sti
	xor	dx,dx			; DL should contain the boot drive ;-))
	jmp	short exec9		; jump to the boot image

; Handle mode 2. This means just to assign a new stack, and then
; push the two parameters onto it before calling the new image.

exec1:	getarg	es,2			; get the execution pointer
	getarg	bx,1			; into ES:BX
	getarg	ds,4			; get the header pointer
	getarg	si,3			; into DS:SI
	getarg	cx,6			; get the parameter pointer
	getarg	di,5			; into CX:DI
	cli
	mov	ax,OLDINTS
	mov	sp,VECTSIZE		; use the old interrupt table as
	mov	ss,ax			; the new stack
	sti
	push	cx
	push	di			; push the parameter pointer
	push	ds
	push	si			; push the header pointer

; Jump to the boot image. The pointer is in ES:BX.

exec9:	assume	ss:nothing
	push	cs
	mov	ax,offset cgroup:exech	; push the return pointer
	push	ax
	push	es			; jump to new image
	push	bx
	retf

exech:	mov	ax,LOWMEM		; fatal() needs correct data segment
	mov	ds,ax			; for prnchr routine!
	jmp	_fatal

_exec_image	endp


;
;**************************************************************************
;
text_end

	end

